/* $Id: dvi.c,v 1.2 2004/11/26 11:29:07 zlb Exp $ */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <assert.h>

#include "common.h"

#if defined(WIN32) && defined(_DEBUG)
#  define DEBUG
#endif

#if defined(WIN32) || defined(GO32)
#  define STRICMP(s1, s2)	stricmp(s1, s2)
#  define NCMP(s1, s2)		strnicmp(s1, s2, strlen(s2))
#else
#  define STRICMP(s1, s2)	strcmp(s1, s2)
#  define NCMP(s1, s2)		strncmp(s1, s2, strlen(s2))
#endif

typedef unsigned char byte;

/* DVI command bytes */
#define dvi_set_char_0    0
#define dvi_set_char_127  127

#define dvi_set1          128
#define dvi_set4          131

#define dvi_put1          133
#define dvi_put4 	  136

#define dvi_bop           139

#define dvi_push          141
#define dvi_pop           142
#define dvi_right4        146
#define dvi_down4         160

#define dvi_fnt_num_0     171
#define dvi_fnt_num_63    234

#define dvi_fnt1          235
#define dvi_fnt4          238

#define dvi_xxx1          239
#define dvi_xxx4          242

#define dvi_fnt_def1      243
#define dvi_fnt_def4      246

#define dvi_pre           247
#define dvi_post          248
#define dvi_post_post     249

#define dvi_undef	  250

static byte CommLen[dvi_post_post-dvi_set1+1] = {
/* The following table contains the lengths in bytes of
   the parameter of DVI commands between "set1" and "post_post" */
/* set1 */	1,  2,  3,  4,  8,  1,  2,  3,  4,  8,  0, 44,  0,  0,  0,
/* right1 */	1,  2,  3,  4,  0,  1,  2,  3,  4,  0,  1,  2,  3,  4,
/* down1 */	1,  2,  3,  4,  0,  1,  2,  3,  4,  0,  1,  2,  3,  4,
/* fnt_num_0 */	0,  0,  0,  0,  0,  0,  0,  0,
		0,  0,  0,  0,  0,  0,  0,  0,
		0,  0,  0,  0,  0,  0,  0,  0,
		0,  0,  0,  0,  0,  0,  0,  0,
		0,  0,  0,  0,  0,  0,  0,  0,
		0,  0,  0,  0,  0,  0,  0,  0,
		0,  0,  0,  0,  0,  0,  0,  0,
		0,  0,  0,  0,  0,  0,  0,  0,
/* fnt1 */      1,  2,  3,  4,  0,  0,  0,  0,  0,  0,  0,  0,
/* pre */	0,  28, 5};

static byte *buffer = NULL;
static size_t bufferlen = 0;

static unsigned long
read_int(int bytes, FILE *f_in)
{
    unsigned long val;
    int i;
    byte b;

    val = 0;
    for (i = 0; i < bytes; i++)
	val = val * 256 + (b = (byte)(fgetc(f_in)));

    return val;
}

static void
write_int(long val, int bytes, FILE *f)
{
    unsigned long div;
    int i;

    div = 1L << ((bytes - 1) * 8);
    for (i = 0; i < bytes; i++) {
	fputc((byte)(val / div), f);
	val %= div;
	div = div / 256;
    }
}

int
fnmap_comp(const void *e1, const void *e2)
{
    return fnstrcmp(((fnmap_t *)e1)->temp_fn, ((fnmap_t *)e2)->temp_fn);
}

int
patch_dvi_file(const char *infile, const char *outfile,
		fnmap_t *fnmap, size_t fnmap_n)
/* fnmap[] must be a sorted list in order to use bsearch() */
{
    byte cmd;
    int i;
    long ll, last_loc = -1;
    FILE *f_in = NULL, *f_out = NULL;

    assert(sizeof(long) >= 4);

    if (buffer != NULL)
	free(buffer);
    buffer = malloc(bufferlen = 2048);
    if (buffer == NULL) {
mem_err:
	fprintf(stderr, "patch_dvi_file: error: cannot allocate memory.");
	if (f_in != NULL)
	    fclose(f_in);
	if (f_out != NULL)
	    fclose(f_out);
	return 1;
    }

    if ((f_in = fopen(infile, "rb")) == NULL) {
	fprintf(stderr, "patch_dvi_file: cannot read from \"%s\".\n", infile);
	free(buffer);
	buffer = NULL;
	bufferlen = 0;
	return 1;
    }
    setvbuf(f_in, NULL, _IOFBF, 32*1024);


    /* Read DVI file header */
    fread(buffer, 15, 1, f_in);
    if (buffer[0] != dvi_pre || buffer[1] != 2) {
	fprintf(stderr, "patch_dvi_file: invalid dvi file \"%s\"!\n", infile);
	fclose(f_in);
	free(buffer);
	buffer = NULL;
	bufferlen = 0;
	return 2;
    }

    if ((f_out = fopen(outfile, "w+b")) == NULL) {
	fprintf(stderr, "patch_dvi_file: cannot write to \"%s\".\n", outfile);
	fclose(f_in);
	free(buffer);
	buffer = NULL;
	bufferlen = 0;
	return 3;
    }
    setvbuf(f_out, NULL, _IOFBF, 32*1024);

    i = buffer[14];
    fread(buffer+15, i, 1, f_in);
#ifdef DEBUG
    buffer[15 + i] = '\0';
    fprintf(stderr, "'%s'\n", buffer+15);
#endif

    fwrite(buffer, 1, i + 15, f_out);

    while (True) {
	if ((i = fgetc(f_in)) == EOF) {
	    fprintf(stderr, "patch_dvi_file: warning: unexpected EOF.\n");
	    break;
	}
	
	cmd = (byte)i;

	if (cmd > dvi_undef) {
	    fprintf(stderr, "patch_dvi_file: warning: invalid cmd byte: %d\n",
			    cmd);
	    fputc(cmd, f_out);
	    continue;
	}

	if (cmd == dvi_post_post) {
	    fputc(cmd, f_out);
	    fread(buffer, 5, 1, f_in);
	    write_int(last_loc, 4, f_out);
	    fputc(buffer[4], f_out);
	    break;
	}
	
	if (cmd < dvi_set1) {
	    fputc(cmd, f_out);
	    continue;
	}
	
	switch (cmd) {
	   case dvi_bop:
		fread(buffer, 4, 11, f_in);
		ll = ftell(f_out);
		fputc(cmd, f_out);
		fwrite(buffer, 4, 10, f_out);
		write_int(last_loc, 4, f_out);
		last_loc = ll;
		break;
	   case dvi_post:
		fread(buffer, 4, 7, f_in);
		ll = ftell(f_out);
		fputc(cmd, f_out);
		write_int(last_loc, 4, f_out);
		fwrite(buffer + 4, 4, 6, f_out);
		last_loc = ll;
		break;
	    case dvi_xxx1: 
	    case dvi_xxx1 + 1:
	    case dvi_xxx1 + 2:
	    case dvi_xxx1 + 3:
		ll = read_int(cmd - dvi_xxx1 + 1, f_in);
		if (bufferlen <= ll) {
		    free(buffer);
		    buffer = malloc(bufferlen = ll + 1024);
		    if (buffer == NULL)
			goto mem_err;
		}
		fread (buffer, 1, (size_t)ll, f_in);
		buffer[(size_t)ll] = '\0';

		if (memcmp(buffer, "src:", 4) == 0) {
		    byte *p = buffer + 4;
		    Boolean have_space = 0;

		    while (isspace(*p)) p++;
		    /* skip line number */
		    while (isdigit(*p)) p++;
		    /* skip possible ":col number" */
		    if (*p == ':')
		      while (isdigit(*(++p)));
		    /* skip optional space(s) */
		    if (isspace(*p)) {
		      have_space = 1;
		      while (isspace(*p)) p++;
		    }

		    if (!fnstrncmp(p, "ctextemp_", 9)) {
			fnmap_t key = {NULL, NULL}, *m;
			key.temp_fn = p + 9;
			m = bsearch(&key, fnmap, fnmap_n, sizeof(fnmap_t), 
					fnmap_comp);
			if (m != NULL) {
			    size_t len = strlen(m->orig_fn);
			    size_t len1 = ll - (p - buffer);

			    /* insert a space if fn begins with a digit */
			    if (!have_space && isdigit(*(m->orig_fn))) {
				len1--;
				*(p++) = ' ';
			    }

			    if (len == len1) {
				memcpy(p, m->orig_fn, len);
			    }
			    else if (len < len1) {
				memcpy(p, m->orig_fn, len);
				strcpy(p + len, p + len1);
			    }
			    else {
				if (ll + (len - len1) >= sizeof(buffer)) {
				    /* should never happen */
				    size_t lp = p - buffer;
				    bufferlen = ll + (len - len1);
				    buffer = realloc(buffer, bufferlen);
				    if (buffer == NULL)
					goto mem_err;
				    p = buffer + lp;
				}
				memmove(p + len - len1, p, len1 + 1);
				memcpy(p, m->orig_fn, len);
			    }
			    ll = ll + len - len1;
			}
		    }
		}

		if (ll < 256)
		    i = 1;
		else if (ll < 256 * 256)
		    i = 2;
		else if (ll < 256 * 256 * 256)
		    i = 3;
		else
		    i = 4;
		fputc(dvi_xxx1 + i - 1, f_out);
		write_int(ll, i, f_out);
		fwrite(buffer, 1, (size_t)ll, f_out);
		break;
	    case dvi_fnt_def1:
	    case dvi_fnt_def1 + 1:
	    case dvi_fnt_def1 + 2:
	    case dvi_fnt_def1 + 3:
		fputc(cmd, f_out);
		/* font no, cksum, ssize, dsize */
		fread (buffer, cmd - dvi_fnt_def1 + 1 + 3 * 4, 1, f_in);
		fwrite(buffer, cmd - dvi_fnt_def1 + 1 + 3 * 4, 1, f_out);
		/* font name */
		fread(buffer, 1, 2, f_in);
		i = buffer[0] + buffer[1];
		fread(buffer + 2, 1, i, f_in);
		fwrite(buffer, i + 2, 1, f_out);
		break;
	    default:
		fputc(cmd, f_out);
		ll = CommLen[cmd - dvi_set1];
		if (ll > 0) {
		    fread (buffer, (size_t)ll, 1, f_in);
		    fwrite(buffer, (size_t)ll, 1, f_out);
		}
		break;
	}
    }
    
    fclose(f_in);
    f_in = NULL;

    i = (int)(ftell(f_out) & 3);
    i = i ? 8 - i : 4;
    memset(buffer, 223, i);
    fwrite(buffer, i, 1, f_out);
    fclose(f_out);
    f_out=NULL;

    free(buffer);
    buffer = NULL;
    bufferlen = 0;

    return 0;
}

#ifdef TEST

/* fnmap[] must be a sorted list in order to use bsearch() */
static fnmap_t fnmap[] = {
    {"README.tex", "README.tex"},
    {"README.tmp", "README.toc"}
};
#define fnmap_n (sizeof(fnmap)/sizeof(fnmap[0]))

int
main(int argc, char *argv[])
{
    if (argc != 3) {
	fprintf(stderr, "Usage: %s infile.dvi outfile.dvi\n", argv[0]);
	exit(1);
    }

    return patch_dvi_file(argv[1], argv[2], fnmap, fnmap_n);
}
#endif
