/* $Id: fontdrvc.c,v 1.2 2004/11/26 11:41:32 zlb Exp $ */

/* This module provides access to CCT fonts (of types 0, 1, 2). */

#undef USE_IOPACK

typedef unsigned char byte;

#include <stdio.h>
#include <setjmp.h>
#include <stdlib.h>
#include <string.h>

#include "encoding.h"
#include "proto.h"

#ifdef _Windows
#  include "win32def.h"
#  include "def.h"		/* to catch redifinition of 'far' */
#else
#  undef USE_IOPACK
#endif

#ifdef USE_IOPACK
#  include "iopack.h"
#endif

#define FAR	huge
#define SHORT	short		/* int. type for saving cut points */

/*#if sizeof(int)==4*/
#if defined(UNIX) || defined(GO32) || defined(WIN32)
#  define far
#  define farmalloc malloc
#  define near
#  define huge
#  define farfree   free
#  define long int
#  undef GRAPHICS
#else
/*#  define GRAPHICS*/
#endif

#ifndef TEST
#  undef GRAPHICS
#endif

#ifdef GRAPHICS
#  include <graphics.h>
#  include <bios.h>
#endif

/*============================== local functions =============================*/

/* Three font formats are supported */
enum {
  TYPE_1_FONT,		/* uncompressed (.FNT), id==0x0001fdfdl */
  TYPE_2_FONT,		/* fixed record length (PZ), id=0x0002fdfdl */
  TYPE_3_FONT		/* compressed (.PPS), id=0x0003fdfdl */
};

/* Error codes */
#define FILE_IO_ERROR		1	/* IO error */
#define FONT_TYPE_ERROR		4	/* Invalid font file */
#define NOT_OPEN_ERROR		8	/* font not open */
#define CHARNO_ERROR		16	/* invalid character no. */
#define SIZE_ERROR		32	/* invalid output width or height */
#define BUFFER_ERROR		64	/* insufficient work buffer */
#define DATA_LEN_ERROR		256	/* extra data found */
#define DATA_ERROR		512	/* extra data found */

#ifdef USE_IOPACK
  static bytefile file_in={NULL, NULL, 0, 0l, -1};
  static bytefile *f_in = &file_in;
#else
  static FILE *f_in=NULL;
#endif

static int width,height,font_type,nbrofchars,reclen=0;
static unsigned buffer_size;
static SHORT far *buffer;
static long far *ptrtab;

static jmp_buf jmp_env;

static int  orientation,datalen;
static int  bm_w,bm_h,bm_wb,bm_w0,bm_h0;
static byte huge *matrix;

static int columns;

static int coordx,coordy;

/* the Huffman coding tree */
static struct tnode {
  byte lchar,rchar;   /* left/right character */
  byte left, right;  /* pointers to the two children */
} tree_table[256];

static byte mask_save,byte_save;

#ifdef TEST
  static int lineno=0;
#  define ErrExit(errno) {        \
    lineno=__LINE__;              \
    longjmp(jmp_env,errno);       \
  }
#else
#  define ErrExit(errno) longjmp(jmp_env,errno);
#endif

#ifdef USE_IOPACK
  #define fread(buff,size,count,f) (gbytes(f,buff,(unsigned)(size)*(count)) ? 0:(count))
  #define fgetc(f)	gbyte(f)
  #define fseek(f, pos, where)	seekpos(f, pos)
#endif

static byte near get_byte0 _VOID
{
  int b;

  if ((b=fgetc(f_in))==EOF) ErrExit(FILE_IO_ERROR);
  if (--datalen<0) ErrExit(DATA_LEN_ERROR);
  return (byte)b;
}

static int near get_byte _VOID
{
  register byte p,p1,f;

  if (font_type!=TYPE_3_FONT) {
    p=get_byte0(); DECODING(p);
    return(p);
  }

  p=1;
  do {
    if (!mask_save) {
      byte_save=get_byte0();
      mask_save=128;
    }
    f=byte_save & mask_save;
    mask_save>>=1;

    if (!f) {
      if ((p1=tree_table[p].left)==0) return(tree_table[p].lchar);
    } else {
      if ((p1=tree_table[p].right)==0) return(tree_table[p].rchar);
    }
    p=p1;
  } while (1);
}

static int near bezier_eval _P3(int *,p, int,t0, int,t1)
/* returns the value of the Bezier function with guiding points p[0],...,p[3]
   at t=t0/t1 */
{
  register int i,n,r;
  long ll;
  int pwork[4],rwork[4];

  n=3;

  if (!t0) return(p[0]); else if (t0==t1) return(p[n]);

  r=(t1+1)/2;

  for (i=0;i<n;i++) {
    ll= (t1-t0)*(long)p[i]+t0*(long)p[i+1];
    pwork[i]=(int)((ll+r)/t1); rwork[i]=(int)(ll-pwork[i]*t1);
  }

  while (--n) for (i=0;i<n;i++) {
    ll= (t1-t0)*(long)pwork[i]+t0*(long)pwork[i+1]+
       ((t1-t0)*(long)rwork[i]+t0*(long)rwork[i+1]+r)/t1;
    pwork[i]=(int)((ll+r)/t1); rwork[i]=(int)(ll-pwork[i]*t1);
  }

  return(pwork[0]);
}

static int near bezier_points _P2(int *,px, int *,py)
/* Estimates number of points needed to approximate a cubic Bezier
   polynomial by a peacewise linear function (see formulae below).
   Returns # of points */
{
  int d,d1,d2;

  d=abs(px[0]-px[1]-px[1]+px[2]);
  d1=abs(px[1]-px[2]-px[2]+px[3]); if (d<d1) d=d1;
  d1=abs(py[0]-py[1]-py[1]+py[2]); if (d<d1) d=d1;
  d1=abs(py[1]-py[2]-py[2]+py[3]); if (d<d1) d=d1;
  /* # of points = sqrt(d*3/(4*eps)) = sqrt(d*3/2) with eps=0.5 */
  d1=(d+d+d+1)>>1;
  if (d1<=1) return(1);
  /* compute sqrt(d1) by Newton method */
  d=(d1+1)>>1;
  do {
    d2=(d+d1/d+1)>>1;
    if (d2==d) break;
    d=d2;
  } while (1);
  return(d+1);
}

static int near get_coord _VOID
/* get next coordinate on the current outline.
   returns:  -1 if end of data
             1  if end of outline
             0  otherwise */
{
  register int c,dx,dy;
  static int x,y,x0,y0,ip,np,px[4],py[4],multi_line_flag=0, curve_flag=0;

  if (multi_line_flag) {	/* within a multi-lineto command */
multi_lineto:
    dx=get_byte(); dy=get_byte();
    if (!dx && !dy) {
      multi_line_flag=0;
      goto next_command;
    }
    x0+=dx; if (x0>=width) x0-=width;
    y0+=dy; if (y0>=height) y0-=height;
    x=(int)((x0*(long)bm_w0)/width);
    y=(int)((y0*(long)bm_h0)/height);
    if (!orientation) {
      coordx=x; coordy=y;
    } else {
      coordy=x; coordx=y;
    }
    return 0;
  }

  if (curve_flag) {			/* within a bezier curve */
curveto:
    if (!orientation) {
      coordx=bezier_eval(px,ip,np);
      coordy=bezier_eval(py,ip,np);
    } else {
      coordy=bezier_eval(px,ip,np);
      coordx=bezier_eval(py,ip,np);
    }
    if (++ip>np) curve_flag=0;
    return 0;
  }

next_command:
  c=get_byte();
  switch (c) {
    case 0:		/* moveto */
      np=x0=y0=0;
      /* falls into next case */
    case 1:		/* lineto */
      x0+=get_byte(); if (x0>=width) x0-=width;
      y0+=get_byte(); if (y0>=height) y0-=height;
      x=(int)((x0*(long)bm_w0)/width);
      y=(int)((y0*(long)bm_h0)/height);
      if (!orientation) {
        coordx=x; coordy=y;
      } else {
        coordy=x; coordx=y;
      }
      return 0;
    case 2:		/* multi-lineto */
      multi_line_flag=1;
      goto multi_lineto;
    case 3:		/* curveto */
      px[0]=x; py[0]=y;
      for (c=1; c<4; c++) {
        x0+=get_byte(); if (x0>=width) x0-=width;
        y0+=get_byte(); if (y0>=height) y0-=height;
        x=(int)((x0*(long)bm_w0)/width);
        y=(int)((y0*(long)bm_h0)/height);
        px[c]=x; py[c]=y;
      }
      np=bezier_points(px,py);
      if (!np) goto next_command;
      curve_flag=ip=1;
      goto curveto;
    case 255:		/* end of data */
      return -1;
    case 254:		/* end of outline */
      return 1;
    default:
      ErrExit(DATA_ERROR);
  }

  return np;
}

static void near save_cut _P2(int,x, int,y)
{
  register int n;
  SHORT far *p;

  if ((n=++(*(p=buffer+y*columns)))<columns) p[n]=(SHORT)x;
  else ErrExit(BUFFER_ERROR);
}

static byte mask[]={0,1,3,7,15,31,63,127,255};  /* mask[i]=(1<<i)-1 */

static byte FAR *setbits _P3(byte FAR *,p, int *,a, int,n)
{
  register int k,i,j;
  if ((i=*a+n)<=8) {
    *a=i;
    j=mask[n]; if ((i=8-i)!=0) j<<=i;
    *p|=(byte)(j & 255);
    if (*a>7) {p++; *a=0;}
  }
  else {
    if (*a) {*(p++)|=mask[i=(8-(*a))]; n-=i;}
    j=n>>3;
    for (k=0;k<j;k++) *(p++)=255;
    if (n&=7) {
      *p|=(byte)(mask[n]<<(8-n));
      *a=n;
    }
    else *a=0;
  }
  return(p);
}

static void near hline _P3(register int,x1, register int,x2, int,y)
{
  int x;
  long ll;

  x=x1 & 7; ll=y*(long)bm_wb+(x1>>3);
  setbits((byte FAR *)(matrix+ll),&x,x2-x1+1);
}

static void near cut_seg _P4(int,x1, int,y1, int,x2, int,y2)
{
  register int x,y,dx,dy,l;

  if (y1>y2) {
    x=x1; x1=x2; x2=x; y=y1; y1=y2; y2=y;
  }

  dx=x2-x1; dy=y2-y1;

  if (!dx) {
    if (y1>y2) {y=y1; y1=y2; y2=y;}
    for (y=y1+1;y<y2;y++) save_cut(x1,y);
    return;
  }

  x=x1; l=dy>>1;

  if (dx>0) {
    for (y=y1+1;;y++) {
      l+=dx;
      if (l>=dy) {
        x+=l/dy; l%=dy;
      }
      hline(x1,x,y-1);  x1=x;
      if (y>=y2) break; else save_cut(x,y);
    }
  } else {
    dx=-dx;
    for (y=y1+1;; y++) {
      l+=dx;
      if (l>=dy) {
        x-=l/dy; l%=dy;
      }
      hline(x,x1,y-1); x1=x;
      if (y>=y2) break; else save_cut(x,y);
    }
  }
}

static void near shellsort _P2(int,n, SHORT far *,a)
{
  register SHORT temp;
  register int gap,j,j1,i;

  for (gap=n>>1; gap>0; gap>>=1)
    for (i=gap; i<n; i++) {
      for (j=i-gap,j1=i; j>=0 && a[j]>a[j1]; j1=j,j-=gap) {
	temp=a[j1]; a[j1]=a[j]; a[j]=temp;
      }
    }
}

static void near sort _VOID
{
  SHORT far *p;
  register int y;

  p=buffer;
  for (y=0;y<bm_h;y++) {
    shellsort(*p,p+1);
    p+=columns;
  }
}

static void near link _VOID
{
  SHORT far *p,far *p0;
  register int x1,x2;
  int i,n,x,y;
  byte FAR *mat;

  p=buffer; mat=(byte FAR *)matrix;
  for (y=0;y<bm_h;y++) {
    n=*p;
    if (n & 1) {
      ErrExit(-1);
    }
    if (n) {
      p0=p+1;
      x1=*(p0++);
      for (i=1;i<n;i++) {
        x2=*(p0++);
        if (i & 1) {
          x=x1 & 7;
          setbits(mat+(x1>>3),&x,x2-x1+1);
        }
        x1=x2;
      }
    }
    p+=columns;
    mat=(byte far *)(((byte huge *)mat)+bm_wb);
  }
}

/*============================= Interface functions ==========================*/

int pkfunc_init _P3(void far *,fn0, unsigned,buffer_size0, void far *,ptrtab0)
/* Open the font whose name is pointed by "fn" and init. internal
   variables. Returns 0 if successful otherwise returns the error
   code */
{
  register int i,j;
  byte s[1024],*p, far *fn;

  buffer_size=buffer_size0;
  ptrtab=(long far *)ptrtab0;

  /* open font file */
  for (fn=(byte *)fn0, i=0; i<1023; i++) {
    if (!fn[i]) break;
    s[i]=fn[i];
  }
  s[i]='\0';
#ifdef USE_IOPACK
  if (f_in->buff==NULL) if (bfalloc(f_in, 512)) return(FILE_IO_ERROR);
  if (openin("", s, f_in, -1)) return(FILE_IO_ERROR);
#else
  if ((f_in=fopen(s,"rb"))==NULL) return(FILE_IO_ERROR);
  if (setvbuf(f_in, NULL, _IOFBF, 512)) goto io_error;
#endif

  /* read file heading */
  if (fread(p=s,12,1,f_in)!=1) {
io_error:
    i=FILE_IO_ERROR;
error:
#ifdef USE_IOPACK
    closein(f_in); free(f_in->buff); f_in->buff=NULL;
#else
    fclose(f_in); f_in=NULL;
#endif
    return(i);
  }

  /* determine type of input font */
  if (p[0]!=0xfd || p[1]!=0xfd || p[3]) {
font_err:
    i=FONT_TYPE_ERROR; goto error;
  }

  switch (p[2]) {
    case 1:	font_type=TYPE_1_FONT; break;
    case 2:	font_type=TYPE_2_FONT; break;
    case 3:	font_type=TYPE_3_FONT; break;
    default:	goto font_err;
  }

  /* get nbr of characters */
  p+=4;
  nbrofchars=(p[3]*256+p[2])-(p[1]*256+p[0])+1;
  if (nbrofchars<=0) goto font_err;

  /* get width & height */
  p+=4;
  width=p[1]*256+p[0];
  height=p[3]*256+p[2];
  if (width<=0 || height<=0) goto font_err;

  if (font_type==TYPE_2_FONT) {
    /* read record length */
    if (fread(s,2,1,f_in)!=1) goto io_error;
    reclen=s[1]*256+s[0];
  } else {
    /* get table of pointers */
    for (j=0;j<=nbrofchars;j++) {
      if (fread(s,4,1,f_in)!=1) goto io_error;
      ptrtab[j]=s[0]+256l*(s[1]+256l*(s[2]+256l*s[3]));
    }
    if (font_type==TYPE_3_FONT) {
      /* read huffman encoding tree */
      if (fread(tree_table,sizeof(struct tnode),1,f_in)!=1) goto io_error;
      j=tree_table[0].lchar;
      if (j<0 || j>=256) goto font_err;
      if (j) {
        if (j!=(int)fread(tree_table+1,sizeof(struct tnode),j,f_in)) goto io_error;
      }
    }
  }

#ifdef TEST
  fprintf(stderr,"Font type: %s.\n",font_type==TYPE_1_FONT ? "FNT":
                  (font_type==TYPE_2_FONT ? "PZ":"PPS"));
  fprintf(stderr,"Number of chars: %d\n",nbrofchars);
  fprintf(stderr,"Width: %d\n",width);
  fprintf(stderr,"Height: %d\n",height);
#endif

  return(0);
}

void pkfunc_close _VOID
/* close the font */
{
#ifdef USE_IOPACK
  if (f_in->fd!=-1) {
    closein(f_in);
  }
  if (f_in->buff!=NULL) {
    free(f_in->buff); f_in->buff=NULL;
  }
#else
  if (f_in!=NULL) {
    fclose(f_in); f_in=NULL;
  }
#endif
}

int pkfunc _P6(void huge *,bmbuffer, int,charno, int,charwidth, int,charheight,
               int,mode, void far *,work)
/* get bit-map data for te character "charno" in buffer.
   "mode" provides some options:
   bit 0:	!=0 ==> transpose the character
   bit 1-2:	Smoothing level (not used here).
   returns 0 if successful otherwise returns an error code

   Only bit 0 of "mode" is meaningful! */
{
  register int xl, xr, i, n;
  long ll;
#if !defined(UNIX) && !defined(GO32)
  unsigned u;
  byte FAR *fp,huge *hp;
#endif

  int x0,y0,x1,y1,x2,y2, x0_save, y0_save, x1_save, y1_save;
  int dy1,dy2;

  /* setup return point when error occurs */
  if ((i=setjmp(jmp_env))!=0) {
#ifdef TEST
    fprintf(stderr,"Error on line no. %d, charno=%d\n",lineno,charno);
#endif
    return(i);
  }

  matrix=bmbuffer;
  buffer=(SHORT far *)work;
  orientation=mode&1;
  if (!orientation) {
    bm_w=charwidth; bm_h=charheight;
  } else {
    bm_w=charheight; bm_h=charwidth;
  }
  bm_wb=(bm_w+7)/8;
  bm_w0=charwidth; bm_h0=charheight;

  if (f_in==NULL) ErrExit(NOT_OPEN_ERROR);
  if (charno<0 || charno>=nbrofchars) ErrExit(CHARNO_ERROR);
  if (bm_w<=0 || bm_h<=0) ErrExit(SIZE_ERROR);

  /* seek to data position */
  if (font_type==TYPE_2_FONT) {
    ll=16l+reclen*(long)charno;
    datalen=reclen;
  } else {
    ll=ptrtab[charno];
    datalen=(int)(ptrtab[charno+1]-ll);
  }
  if (fseek(f_in,ll,SEEK_SET)) ErrExit(FILE_IO_ERROR);

  /* clear output matrix */
#if !defined(UNIX) && !defined(GO32)
  ll=bm_wb*(long)bm_h;
  hp=(byte huge *)bmbuffer;
  while (ll) {
    u=(ll>32760l) ? 32760:(int)ll;
    fp=(byte FAR *)hp;
    for (i=0;i<(int)u;i++) *(fp++)=0;
    ll-=u; hp+=u;
  }
#else
  memset(bmbuffer,0,bm_wb*bm_h);
#endif

  /* setup work buffer */
  columns=buffer_size/(sizeof(SHORT)*bm_h);
  if (!columns) ErrExit(BUFFER_ERROR);
  /* init. cut-points counters */
  for (i=0;i<bm_h;i++) buffer[i*columns]=0;

  /* process outlines */
  if (font_type==TYPE_3_FONT) {
    mask_save=0;	/* compressed font */
  } else {
    ENCODING_INIT(charno);
  }
  while (1) {
next_outline:
    i=0;
    /* get first point */
    do {
      n=get_coord();
      if (n==-1) break;
    } while (n);
    if (n==-1) break;
    x0_save=x0=coordx; y0_save=y0=coordy;

    /* get second point */
    do {
      n=get_coord();
      if (n==-1) {
        ErrExit(DATA_ERROR);
      } else if (n) {		/* the outline contains a single point */
        hline(x0,x0,y0);
        goto next_outline;	/* pass to next outline */
      } else {
        x1=coordx; y1=coordy;
      }
    } while (x1==x0 && y1==y0);
    x1_save=x1; y1_save=y1;

    dy1=y1-y0; xl=xr=x1;

    while (i<3) {
next_point:
      if (i==1) {
        x2=x0_save; y2=y0_save;
        i=2;
      } else if (i==2) {
        x2=x1_save; y2=y1_save;
        i=3;
      } else do {
        n=get_coord();
        if (n==1) {
          i=1; goto next_point;
        } else if (n==-1) {
          ErrExit(DATA_ERROR);
        }
        x2=coordx; y2=coordy;
      } while (x2==x1 && y2==y1);
      dy2=y2-y1;
      if (!dy1) {
        if (xl>x1) xl=x1; else if (xr<x1) xr=x1;
 	if (dy2) {
          hline(xl,xr,y1);
          if (dy2<0) {
            save_cut(x1,y1);
          }
          cut_seg(x1,y1,x2,y2);
        }
      } else if (!dy2) {
        xl=xr=x1; if (dy1>0) save_cut(x1,y1);
      } else if ((dy1>0)==(dy2>0)) {
        save_cut(x1,y1); cut_seg(x1,y1,x2,y2);
      } else {
        cut_seg(x1,y1,x2,y2);
        if (x0<x1) x0=x1-(abs(x1-x0)/abs(dy1)+1)/2; else x0=x1;
        if (x1<x2) y0=x1+(abs(x2-x1)/abs(dy2)+1)/2; else y0=x1;
        hline(x0,y0,y1);
      }
      x0=x1; y0=y1; x1=x2; y1=y2;
      dy1=dy2;
      (void)y0;
    }
    if (dy1==0) {
      if (xl>x1) xl=x1; else if (xr<x1) xr=x1;
      hline(xl,xr,y1);
    }
  }
  sort();
  link();
  return(/* datalen ? DATA_LEN_ERROR:*/0);
}

/*=================================== Test ===================================*/

#ifdef TEST
#if !defined(UNIX) && !defined(GO32)
#  include <malloc.h>
#endif

#define BUFFERSIZE 	0xfff0
#define WIDTH		256
#define HEIGHT		256
#define MODE            0

int main _P2(int,argc, char **,argv)
{
  void far *ptrtab, far *buffer;
  byte *matrix;
  int i,code,first,last;
#ifdef GRAPHICS
  int graphdriver=VGA, graphmode=VGAHI, scrwb, scrheight, posx, posy;
  register int j,k,wb=(WIDTH+7)/8;
  byte far *scr=(void far *)0xa0000000l;

  initgraph(&graphdriver, &graphmode, "c:\\tc");
  scrwb=(getmaxx()+8)/8;
  scrheight=getmaxy()+1;
  posx=0; posy=0;
#else
  int j, k, l, b, minx, miny, maxx, maxy;
#endif

  /* check for size of tree_table[] */
  i=sizeof(struct tnode);
  if (i!=4) {
    fprintf(stderr,"Size of struct tnode is not 4!\n");
    return 1;
  }

  if (argc < 2 || argc > 4) {
    fprintf(stderr,"Usage: fontdrvc font_file [firstchar [lastchar] ...]\n");
    fprintf(stderr,"(default: firstchar=0, lastchar=# of chars - 1)\n");
    return 2;
  }

  ptrtab=farmalloc(4000l*4l);
  buffer=farmalloc((long)BUFFERSIZE);
  matrix=malloc(HEIGHT*((WIDTH+7)/8));
  if (ptrtab==NULL || buffer==NULL || matrix==NULL) {
    fprintf(stderr,"Memory allocation error!\n");
    return 3;
  }

  code=pkfunc_init(argv[1],(unsigned)BUFFERSIZE,ptrtab);
  if (code) {
    fprintf(stderr,"Error in pkfunc_init(), error no=%d\n", code);
    return 4;
  }

  first = (argc >= 3) ? atoi(argv[2]) : 0;
  last = (argc >= 4) ? atoi(argv[3]) : nbrofchars - 1;

  for (i=first; i<=last; i++) {
    code=pkfunc(matrix,i,WIDTH,HEIGHT,MODE,buffer);
    if (code) {
      fprintf(stderr,"Error in pkfunc(), error no=%d\n", code);
      return 5;
    }
#ifdef GRAPHICS
    for (j=0; j<HEIGHT; j++) for (k=0; k<wb; k++) {
      scr[posx+k+scrwb*(posy+j)]=matrix[j*wb+k];
    }
    posx+=wb;
    if (posx>scrwb-wb) {
      posx=0;
      posy+=HEIGHT;
      if (posy>scrheight-HEIGHT) {
        if ((bioskey(0)&255)==27) goto fin;
        posy=0;
      }
    }
#else  /* GRAPHICS */
    /* Compute the Bounding Box */
    maxx=maxy=0;
    minx=WIDTH-1; miny=HEIGHT-1;
    for (k=0; k<HEIGHT; k++) {
	for (j=0; j<(WIDTH+7)/8; j++) {
	    b = matrix[k*((WIDTH+7)/8)+j];
	    if (b == 0) continue;
	    if (miny > HEIGHT-1-k) miny = HEIGHT-1-k;
	    if (maxy < HEIGHT-1-k) maxy = HEIGHT-1-k;
	    l = 0; 	/* it makes gcc happy. */
	    if      ((b&128) != 0) l = j*8 + 0;
	    else if ((b&64)  != 0) l = j*8 + 1;
	    else if ((b&32)  != 0) l = j*8 + 2;
	    else if ((b&16)  != 0) l = j*8 + 3;
	    else if ((b&8)   != 0) l = j*8 + 4;
	    else if ((b&4)   != 0) l = j*8 + 5;
	    else if ((b&2)   != 0) l = j*8 + 6;
	    else if ((b&1)   != 0) l = j*8 + 7;
	    if (minx > l) minx = l;
	    if      ((b&1)   != 0) l = j*8 + 7;
	    else if ((b&2)   != 0) l = j*8 + 6;
	    else if ((b&4)   != 0) l = j*8 + 5;
	    else if ((b&8)   != 0) l = j*8 + 4;
	    else if ((b&16)  != 0) l = j*8 + 3;
	    else if ((b&32)  != 0) l = j*8 + 2;
	    else if ((b&64)  != 0) l = j*8 + 1;
	    else if ((b&128) != 0) l = j*8 + 0;
	    if (maxx < l) maxx = l;
	}
    }
    printf("char %d, BoundingBox: %d %d %d %d (0 0 %d %d)\n",
		    i, minx, miny, maxx, maxy, WIDTH-1, HEIGHT-1);
#endif /* GRAPHICS */
  }

#ifdef GRAPHICS
fin:
#endif
  pkfunc_close();
#ifdef GRAPHICS
  bioskey(0);
  closegraph();
#endif /* GRAPHICS */

  return 0;
}
#endif /* TEST */
