/* $Id: charcnt.c,v 1.3 2005/03/21 02:21:45 zlb Exp $ */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#if defined(WIN32) && defined(_DEBUG)
#  define DEBUG
#endif

#if defined(WIN32) || defined(GO32)
#  define STRICMP(s1, s2)	stricmp(s1, s2)
#  define NCMP(s1, s2)		strnicmp(s1, s2, strlen(s2))
#else
#  define STRICMP(s1, s2)	strcmp(s1, s2)
#  define NCMP(s1, s2)		strncmp(s1, s2, strlen(s2))
#endif

typedef unsigned char byte;

/* DVI command bytes */
#define dvi_set_char_0    0
#define dvi_set_char_127  127

#define dvi_set1          128
#define dvi_set4          131

#define dvi_put1          133
#define dvi_put4 	  136

#define dvi_bop           139

#define dvi_push          141
#define dvi_pop           142
#define dvi_right4        146
#define dvi_down4         160

#define dvi_fnt_num_0     171
#define dvi_fnt_num_63    234

#define dvi_fnt1          235
#define dvi_fnt4          238

#define dvi_xxx1          239
#define dvi_xxx4          242

#define dvi_fnt_def1      243
#define dvi_fnt_def4      246

#define dvi_pre           247
#define dvi_post          248
#define dvi_post_post     249

#define dvi_undef	  250

#define pk_pre   247
#define pk_post  245
#define pk_no_op 246

static byte CommLen[dvi_post_post-dvi_set1+1] = {
/* The following table contains the lengths in bytes of
   the parameter of DVI commands between "set1" and "post_post" */
/* set1 */	1,  2,  3,  4,  8,  1,  2,  3,  4,  8,  0, 44,  0,  0,  0,
/* right1 */	1,  2,  3,  4,  0,  1,  2,  3,  4,  0,  1,  2,  3,  4,
/* down1 */	1,  2,  3,  4,  0,  1,  2,  3,  4,  0,  1,  2,  3,  4,
/* fnt_num_0 */	0,  0,  0,  0,  0,  0,  0,  0,
		0,  0,  0,  0,  0,  0,  0,  0,
		0,  0,  0,  0,  0,  0,  0,  0,
		0,  0,  0,  0,  0,  0,  0,  0,
		0,  0,  0,  0,  0,  0,  0,  0,
		0,  0,  0,  0,  0,  0,  0,  0,
		0,  0,  0,  0,  0,  0,  0,  0,
		0,  0,  0,  0,  0,  0,  0,  0,
/* fnt1 */      1,  2,  3,  4,  0,  0,  0,  0,  0,  0,  0,  0,
/* pre */	0,  28, 5};

typedef enum {
	FONT_CCT,		/* 'ccfnt*' */
	FONT_CCT_TEMP,		/* 'tmp###' */
	FONT_CCT_DUMMY,		/* 'ccdummy' */
	FONT_CJK_GBK,		/* 'gbk*' */
	FONT_MATH, 		/* TeX Math fonts */
	FONT_OTHER		/* nomal western fonts */
} FontType;

static char *FontTypeDesc[] = {
	"ȫַ:",
	"ȫַ:",
	NULL,
	"ȫַ:",
	"ѧ:",
	"ַ:"
};

static FILE *f_dvi = NULL;

static long	fontno = -1;			/* current fontno */
static char	*fontname = "undefined";	/* current fontname */
FontType	fonttype = FONT_OTHER;

/* list of fonts */
static struct {
    long	fontno;
    char	*fontname;
    FontType	fonttype;
} *fontlist = NULL, *fp;
static int nfonts = 0;

static int counters[FONT_OTHER + 1];

static long
read_number(int size)
{
    long n = 0l;
    int i;

    for (i = 0; i < size; i++)
	n = (n << 8) | (byte)fgetc(f_dvi);

    return n;
}

static void
setput_char(byte ch)
{
#ifdef DEBUG
    fprintf(stderr, "Char %d `%c'\n", ch, (ch >32 && ch < 127) ? ch : ' ');
#endif

    /* Note: two bytes = one Chinese char if fonttype == FONT_CCT */
    counters[fonttype]++;
}

static void
define_font(long no, char *name)
{
    fontlist = realloc(fontlist, sizeof(*fontlist) * (nfonts + 1));
    fp = fontlist + (nfonts++);
    if (fontlist == NULL || (fp->fontname = malloc(strlen(name) + 1)) == NULL) {
	fprintf(stderr, "Memory allocation error.\n");
	exit(4);
    }
    strcpy(fp->fontname, name);
    fp->fontno = no;

    /* determine type of font */
    if (!STRICMP(name, "ccdummy"))
	fp->fonttype = FONT_CCT_DUMMY;
    else if (!NCMP(name, "ccfnt"))
	fp->fonttype = FONT_CCT;
    else if (!NCMP(name, "tmp"))
	fp->fonttype = FONT_CCT_TEMP;
    else if (!NCMP(name, "gbk"))
	fp->fonttype = FONT_CJK_GBK;
    else if (!NCMP(name, "cmmi") || !NCMP(name, "cmsy") || !NCMP(name, "cmex"))
	fp->fonttype = FONT_MATH;
    else
	fp->fonttype = FONT_OTHER;
#ifdef DEBUG
    fprintf(stderr, "Define font %ld, font name = `%s', font type = `%s'\n",
	no, name, FontTypeDesc[fp->fonttype] == NULL ? "NULL" :
		FontTypeDesc[fp->fonttype]);
#endif
}

static void
select_font(long no)
{
    int i;

    if (no == fontno)
	return;
    fontno = no;
    for (i = 0; i < nfonts; i++)
	if (fontlist[i].fontno == fontno) break;
    if (i >= nfonts) {
	fprintf(stderr, "Error: select_font: font %ld undefined.\n", fontno);
	exit(5);
    }
    fontname = fontlist[i].fontname;
    fonttype = fontlist[i].fonttype;

#ifdef DEBUG
    fprintf(stderr, "Select font %ld, name = `%s'\n", fontno, fontname);
#endif
}

int
main(int argc, char *argv[])
{
    byte cmd, buffer[1024];
    int i;
    long ll;

    assert(sizeof(long) >= 4);

    if (argc != 2) {
	fprintf(stderr, "Usage: %s dvifile[.dvi]\n", argv[0]);
	exit(1);
    }

    memset(counters, 0, sizeof(counters));

    strcpy(buffer, argv[1]);
    if ((i = strlen(buffer)) < 4 || strcmp(buffer + i - 4, ".dvi"))
	strcat(buffer, ".dvi");

    printf("DVI ļ: %s\n", buffer);

    if ((f_dvi = fopen(buffer, "rb")) == NULL) {
	fprintf(stderr, "Cannot open dvi file.\n");
	exit(2);
    }
    setvbuf(f_dvi, NULL, _IOFBF, 32*1024);

    /* Read DVI file header */
    fread(buffer, 15, 1, f_dvi);
    if (buffer[0] != dvi_pre || buffer[1] != 2) {
	fprintf(stderr, "Invalid input file!\n");
	fclose(f_dvi);
	exit(3);
    }
    i = buffer[14];
    fread(buffer+15, i, 1, f_dvi);
#ifdef DEBUG
    buffer[15 + i] = '\0';
    fprintf(stderr, "'%s'\n", buffer+15);
#endif

    while (1) {
	if ((i = fgetc(f_dvi)) == EOF) break;
	cmd = (byte)i;
			/* fprintf(stderr, "%ld: %d\n", ftell(f_dvi) - 1, cmd); */
	if (cmd > dvi_post_post) {
	    fprintf(stderr, "Error: invalid command byte %d encountered.\n",
		cmd);
	    break;
	}
	if (cmd == dvi_post)
	    break;
	if (cmd < dvi_set1) {
	    setput_char(cmd);
	    continue;
	}
	if (cmd >= dvi_fnt_num_0 && cmd <= dvi_fnt_num_63) {
	    select_font((long)(cmd - dvi_fnt_num_0));
	    continue;
	}
	switch (cmd) {
	    case dvi_set1:
	    case dvi_set1 + 1:
	    case dvi_set1 + 2:
	    case dvi_set1 + 3:
		setput_char((byte)read_number(cmd - dvi_set1 + 1));
		break;
	    case dvi_put1:
	    case dvi_put1 + 1:
	    case dvi_put1 + 2:
	    case dvi_put1 + 3:
		setput_char((byte)read_number(cmd - dvi_put1 + 1));
		break;
	    case dvi_fnt1:
	    case dvi_fnt1 + 1:
	    case dvi_fnt1 + 2:
	    case dvi_fnt1 + 3:
		select_font(read_number(cmd - dvi_fnt1 + 1));
		break;
	    case dvi_xxx1: 
	    case dvi_xxx1 + 1:
	    case dvi_xxx1 + 2:
	    case dvi_xxx1 + 3:
		fseek(f_dvi, read_number(cmd - dvi_xxx1 + 1), SEEK_CUR);
		break;
	    case dvi_fnt_def1:
	    case dvi_fnt_def1 + 1:
	    case dvi_fnt_def1 + 2:
	    case dvi_fnt_def1 + 3:
		ll = read_number(cmd - dvi_fnt_def1 + 1); /* font no */
		read_number(4);				/* cksum */
		read_number(4);				/* ssize */
		read_number(4);				/* dsize */
		i = (byte)fgetc(f_dvi) + (byte)fgetc(f_dvi);
		fread(buffer, i, 1, f_dvi);		/* font name */
		buffer[i] = '\0';
		define_font(ll, buffer);
		break;
	    default:
		fseek(f_dvi, (long)CommLen[cmd - dvi_set1], SEEK_CUR);
	}
    }
    
    fclose(f_dvi);
    f_dvi = NULL;

    counters[FONT_CCT] /= 2;

    for (i = 0; i <= FONT_OTHER; i++)
	if (FontTypeDesc[i] != NULL && counters[i])
	    printf("%s %d\n", FontTypeDesc[i], counters[i]);

    return 0;
}
