/* $Id: cctmkind.c,v 1.4 2005/04/11 01:27:44 zlb Exp $ */

/* Makeindex for CCT */

typedef unsigned char byte;

#define version "0.0.4"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <unistd.h>

#include "string-utils.h"
#include "roman-number.h"
#include "GBK-utils.h"
#include "cctmkind-ist.h"

#define MESG(a)		if (!quiet) fprintf a
#define ERROR(a)	fprintf a

typedef enum 		{NONE,   RSTART,   REND,   MATCHED} tag_t;
#ifdef DEBUG
static char *tags[] =	{"NONE", "RSTART", "REND", "MATCHED"};
#endif
typedef enum {False, True} Boolean;

typedef struct {
  byte 	*key,  *subkey,  *subsubkey;
  byte  *item, *subitem, *subsubitem;
  byte	*pageno;
  byte	*cmd;
  tag_t	tag;
} entry_t;

static entry_t empty_entry =
	{NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NONE};
static entry_t *entries = NULL;
static size_t entries_n = 0;
static size_t entries_max = 0;

static Boolean quiet = False;
static Boolean remove_blancs = False;
static Boolean letter_ordering = False;
static byte    *start_no = NULL;

static byte buffer[4096];

#ifdef DEBUG
static void
dump_entry(const entry_t *e)
{
    tag_t tag = e->tag;

    MESG((stderr, "[CCTmkind] key=|%s!%s!%s| item=|%s!%s!%s|\n"
		  "  tag=%s cmd=|%s| page=|%s|\n",
		e->key, e->subkey, e->subsubkey, 
		e->item, e->subitem, e->subsubitem, 
		tags[e->tag],
		e->cmd != NULL ? e->cmd : (byte *)"",
		e->pageno));
}

static void
dump_all_entries(void)
{
    int i;

    for (i = 0; i < entries_n; i++) dump_entry(entries + i);
}
#endif

static char *
default_ext(char **fname, const char *ext)
{
    size_t i, n;

    if (*fname == NULL) return NULL;
    /* Hack: don't append extension to, say, /dev/null */
    if (!memcmp(*fname, "/dev/", 5)) return *fname;
   
    i = n = strlen(*fname);
    while (i-- > 0 && (*fname)[i] != '.' && (*fname)[i] != '/'
#ifndef UNIX
	&& (*fname)[i] != '\\' && (*fname)[i] != ':'
#endif
    );
    if (i>=0 && (*fname)[i] == '.') return (*fname);
    (*fname) = realloc(*fname, n + strlen(ext) + 1);
    return strcat((*fname), ext);
}

static Boolean
is_quoted(const char *str, const char *pos)
{
    int i;

    if (pos <= str || *(--pos) != quote)
	return False;

    if (*(--pos) != escape && *pos != quote) 
	return True;

    for (i = 1; pos>str && (*(--pos) == escape || *pos == quote); i++);
    return !(i&1);
}

static char *
find_unquoted_char(byte *str, byte c)
{
    byte *p = str;

    if (str == NULL) return NULL;
    
    while (*p != '\0') {
	if (*p == c && !is_quoted(str, p)) return p;
	if (*p >= 129) if (*(++p) == '\0') break;
	p++;
    }

    return NULL;
}

static void
do_remove_quotes(byte **str)
{
    byte *p = *str;
    size_t l0, l;
	    
    if (p == NULL) return;

    l0 = l = strlen(p);
    while (*p != '\0') {
	if (*p == escape) {
	    if (*(++p) != '\0') p++;
	    continue;
	}
	if (*p == quote) {
	    memmove(p, p+1, (*str + l) - p);
	    l--;
	}
	p++;
    }
    if (l0 != l) *str = realloc(*str, l + 1);
}

static void
do_remove_blancs(byte **str)
/* remove leading and trailing remove_blancs */
{
    byte *p = *str, *q0, *q;

    if (p == NULL) return;
    
    while (isspace(*p)) p++;

    q = q0 = p + strlen(p);
    while (q > p && isspace(*(--q)));
    if (!isspace(*q)) q++;
    if (p > *str || q < q0) {
	*q = '\0';
	p = strdup(p);
	free(*str);
	*str = p;
    }
}

static void
parse_actual(byte **item, byte **key)
{
    byte *p;

    if (*item != NULL) {
	if ((p = find_unquoted_char(*item, actual)) != NULL) {
	    size_t size = strlen(*item) + 1;
	    *p = '\0';
	    *key = strdup(*item);
	    memmove(*item, p + 1, size -= (p + 1) - *item);
	    *item = realloc(*item, size);
	}
	if (*key == NULL) *key = strdup(*item);
    }
    do_remove_quotes(key);
    do_remove_quotes(item);

    if (remove_blancs) {
	do_remove_blancs(key);
	do_remove_blancs(item);
    }

    if (*key != NULL && letter_ordering) {
	Boolean flag = False;
	size_t l = strlen(*key) + 1;
	p = *key;
	while (*p != '\0') {
	    if (isspace(*p)) {
		byte *q = p;
		while (isspace(*(++q)));
		memmove(p, q, l - (q - *key));
		l -= q - p;
		flag = True;
	    }
	    else
		p++;
	}
	if (flag) *key = realloc(*key, l);
    }
}

static int
parse_entry(char *str, byte *pageno, entry_t *e)
{
    byte *p;

    if (pageno == NULL) return 0;
    while (isspace(*pageno)) pageno++;
    /* remove leading and trailing spaces in pageno */
    p = pageno + strlen(pageno);
    while (p > pageno && isspace(*(--p)));
    if (isspace(*p)) return 0;
    p[1] = '\0';

    *e = empty_entry;
    e->pageno = strdup(pageno);

    /* Process '|' */
    if ((p=find_unquoted_char(str, encap)) != NULL) {
	*p = '\0';
	if (p[1] == range_open || p[1] == range_close) {
	    e->tag = *(++p) == range_open ? RSTART : REND;
	}
	if (*(++p) != '\0') e->cmd = strdup(p);
    }
    
    /* Now the item */
    if ((p = find_unquoted_char(str, level)) != NULL) {
	*p = '\0';
	e->item = strdup(str);
	str = p + 1;
	if ((p = strchr(str, level)) != NULL) {
	    *p = '\0';
	    e->subitem = strdup(str);
	    e->subsubitem = strdup(p+1);
	}
	else {
	    e->subitem = strdup(str);
	}
    }
    else {
	e->item = strdup(str);
    }

    /* Extract keys in item, subitem, ... */
    parse_actual(&e->item, &e->key);
    parse_actual(&e->subitem, &e->subkey);
    parse_actual(&e->subsubitem, &e->subsubkey);

    return e->item != NULL;
}

static Boolean
is_digital_number(const byte *n)
{
    while (isspace(*n)) n++;
    if (*n == '+' || *n == '-') n++;
    if (!isdigit(*n)) return False;
    while (isdigit(*n)) n++;
    return *n == '\0';
}

static int
compare_strings(const byte *str1, const byte *str2, Boolean exact)
/* Note: this function (and the other compare_* functions) is called
 * with 'exact' == True when comparing whether two entries have identical
 * items, and with 'exact' == False when called by qsort */
{
    const byte *s1 = str1, *s2 = str2;

    if (s1 == NULL) return s2 == NULL ? 0 : -1;
    if (s2 == NULL) return 1;

    if (exact)
	return strcmp(s1, s2);	/* exact match */

    while (*s1 != '\0' && *s2 != '\0') {
	int c1, c2;

	if (*s1 >= 128 && *s2 >= 128) {
	    if ((c1 = GBK_char_comp(s1, s2)) != 0) return c1;
	    s1 += 2;
	    s2 += 2;
	    continue;
	}
	
	if (*s1 >= 128 || *s2 >= 128)
	    return (*s1 < 128) ? -1 : 1;
	
	/* ASCII ordering: other, digit, alphabet */
	c1 = toupper(*s1);
	c2 = toupper(*s2);

	if (c1 != c2) {
	    if (isalpha(c1)) return isalpha(c2) ? c1 - c2 : 1;
	    if (isalpha(c2)) return -1;

	    if (isdigit(c1)) return isdigit(c2) ? c1 - c2 : 1;
	    if (isdigit(c2)) return -1;

	    return c1 - c2;
	}

	if (*s1 != *s2) return *s1 - *s2;
	
	s1++; s2++;
    }

    /* In case two strings are equal, do a further case sensitive comparison */
    return strcmp(str1, str2);
}

static int
compare_entries(const entry_t *e1, const entry_t *e2, Boolean exact)
{
    int i;

    if (!exact && (i=compare_strings(e1->key, e2->key, exact)) != 0)
	return i;
    if ((i = compare_strings(e1->item, e2->item, exact)) != 0)
	return i;

    if (!exact && (i=compare_strings(e1->subkey, e2->subkey, exact)) != 0)
	return i;
    if ((i = compare_strings(e1->subitem, e2->subitem, exact)) != 0)
	return i;

    if (!exact && (i=compare_strings(e1->subsubkey, e2->subsubkey, exact)) != 0)
	return i;
    return compare_strings(e1->subsubitem, e2->subsubitem, exact);
}

static int
compare_numbers(const byte *p1, const byte *p2)
/* Note: the function should only return -1, 0 or 1 for two numbers
 * of the same type (both decimal, both roman of same case, etc.).
 * In all other cases, the returned value should be always >=2 or <=-2 */
{
    int d1, d2;

    if (p1 == NULL) return p2 == NULL ? 0 : -2;
    if (p2 == NULL) return 2;

    /* check for decimal numbers */
    d1 = is_digital_number(p1);
    d2 = is_digital_number(p2);
    if (d1) return d2 ? atoi(p1) - atoi(p2) : 2;
    if (d2) return -2;

    /* check for roman numbers */
    d1 = roman_number(p1);
    d2 = roman_number(p2);
    if (d1) return d2 ? d1 - d2 : 2;
    if (d2) return -2;

#if defined(__GNUC__) && !defined(__MINGW32__)
#warning FIXME: other numberings (Chinese, etc).
#endif

    return 2 * strcmp(p1, p2);	/* non numbers are not consequent */
}

static int
sort_compare(const void *e1, const void *e2)
{
    int i;
   
    if ((i = compare_entries(e1, e2, False)) != 0)
	return i;

    if ((i = compare_numbers(((entry_t *)e1)->pageno,
			     ((entry_t *)e2)->pageno)) != 0) return i;

    if ((i = ((entry_t *)e1)->tag - ((entry_t *)e2)->tag) != 0) return i;

    if ((i = compare_strings(((entry_t *)e1)->cmd,
			     ((entry_t *)e2)->cmd, False)) != 0) return i;

    /* This will preserve the original order of the two entries when
     * they have identical keys, items, cmd, and pageno */
    return e1 > e2 ? 1 : -1;
}

static Boolean
is_hex(byte c)
{
    return ('0' <= c && c <= '9') || ('a' <= c && c <= 'f');
}

static void
convert(byte *buffer)
/* convert characters of the forms: ^^xx, \CCTSetChar ? ?, etc. */
{
    byte *p = buffer, *q, c, s[3] = "  ";

    while (*p != '\0') {
	if (*p == '\\') {
	    if (!isalpha(*(++p))) {
		if (*(p++) == '\0') return;
		continue;
	    }
	    /* scan TeX command */
	    q = p;
	    while (isalpha(*q)) q++;
	    c = *q;
	    *q = '\0'; 
	    if (!strcmp(p, "verb")) {
		*q = c;
		while (isspace(*q)) q++;
		c = *q;
		while (*(++q) != c) if (*q == '\0') return;
		p = q + 1;
		continue;
	    }
	    else if (!strcmp(p, "CCTSetChar")) {
		*q = c;
		while (isspace(*q)) q++;
		if (*q == '\0') return;
		if (*q == '^' && q[1] == '^' &&
			is_hex(s[0] = q[2]) && is_hex(s[1] = q[3])) {
		    *(p - 1) = strtol(s, NULL, 16);
		    q += 4;
		}
		else {
		    *(p - 1) = *(q++);
		}
		while (isspace(*q)) q++;
		if (*q == '\0') return;
		if (*q == '^' && q[1] == '^' &&
			is_hex(s[0] = q[2]) && is_hex(s[1] = q[3])) {
		    *(p++) = strtol(s, NULL, 16);
		    q += 4;
		}
		else {
		    *(p++) = *(q++);
		}
		memmove(p, q, strlen(q) + 1);
		continue;
	    }
	    else {
		*q = c;
		p = q;
		continue;
	    }
	}
	else if (*p == '^') {
	    if (p[1] != '^') {
		p++;
		continue;
	    }
	    else if (!is_hex(s[0] = p[2]) || !is_hex(s[1] = p[3])) {
		p += 2;
		continue;
	    }
	    /* ^^xx form */
	    *(p++) = strtol(s, NULL, 16);
	    memmove(p, p + 3, strlen(p) - 2);
	    continue;
	}
	else {
	    p++;
	    continue;
	}
    }
}

static void
read_idx(FILE *f_in)
/* loads an .idx file. The current code assumes that each \indexentry command
 * is on a single line and is strictly of the form \indexentry{#1}{#2}
 * (no extra spaces before the opening or after the closing curly braces) */
{
    int i, lineno, n0 = entries_n;
    Boolean newline;
    byte *p, *q, *r;
    entry_t e;

    newline = False;
    lineno = 0;
    while (fgets(buffer, sizeof(buffer), f_in) != NULL) {
	lineno++;
	convert(buffer);
	p = buffer + strlen(buffer) - 1;
	while (p >= buffer && isspace(*p)) p--;
	p[1] = '\0';
	p = buffer;
	while (isspace(*p)) p++;
	if (strncmp(p, keyword, strlen(keyword))) {
	    /* line does not begin with '\indexentry', silently ignored */
	    continue;
	}
	p += strlen(keyword);
	while (isspace(*p)) p++;
	if (*(p++) != arg_open) continue;
	/* find closing '}' */
      	i = 0;
      	for (q = p; i != 0 || *q != arg_close; q++) {
	    if (*q == escape) {q++; continue;}
	    if (*q == arg_open) i++;
	    if (*q == arg_close && --i < 0) break;
	    if (*q == '\0') break;
	} 
	if (i != 0 || *q != arg_close) {
    ignore:
	    if (!newline) {
		newline = True;
		MESG((stderr, "\n"));
	    }
	    MESG((stderr, "[CCTmkind] Warning: invalid input on line %d ignored.\n",
				    lineno));
	    continue;
	}
	*(q++) = '\0';
	if (*(q++) != arg_open) goto ignore;
	i = 0;
      	for (r = q; i != 0 || *r != arg_close; r++) {
	    if (*r == escape) {r++; continue;}
	    if (*r == arg_open) i++;
	    if (*r == arg_close && --i < 0) break;
	    if (*r == '\0') break;
	} 
	if (i != 0 || *r != arg_close) goto ignore;
	*r = '\0';
	/* Now p points to the item string, q points to the page number */
	if (!parse_entry(p, q, &e)) goto ignore;
	if (entries_n >= entries_max) 
	    entries = realloc(entries, (entries_max+=128) * sizeof(entry_t));
	entries[entries_n++] = e;
    }
    MESG((stderr, " %d entries.\n", (int)(entries_n - n0)));
}

static void
append_range(byte **saved, byte *sno, byte *eno, byte *cmd, entry_t *e)
{
    static byte *s = NULL, *delim, *range;
    static size_t s_size = 0;
    size_t size;
    int i = compare_numbers(eno, sno);

    if (i == 1 && suffix_2p[0] != '\0')
	range = suffix_2p;
    else if (i == 2 && suffix_2p[0] != '\0')
	range = suffix_3p;
    else if (suffix_mp[0] != '\0')
	range = suffix_mp;
    else
	range = delim_r;

    if (**saved == '\0') {
	delim = (e->subsubitem != NULL) ? delim_2 :
		 (e->subitem != NULL) ? delim_1 : delim_0;
    }
    else {
	delim = delim_n;
    }

#ifdef DEBUG
    MESG((stderr, "[CCTmkind] append_range: saved=|%s|, sno=|%s|, eno=|%s|, cmd=|%s|\n",
		*saved, sno, eno, cmd));
#endif
   
    size = 16 + strlen(sno) + strlen(eno) + (cmd == NULL ? 0 : strlen(cmd));
    if (size >= s_size) s = realloc(s, s_size += 256);

    if (cmd == NULL) {
	if (i == 0)
	    sprintf(s, "%s%s", delim, sno);
	else
	    sprintf(s, "%s%s%s%s", delim, sno, i < 2 ? delim_n : range, eno);
    }
    else {
	if (i == 0)
	    sprintf(s, "%s%s%s%s%s%s", delim, encap_prefix, cmd, encap_infix,
		sno, encap_suffix);
	else
	    sprintf(s, "%s%s%s%s%s%s%s%s%s%s%s%s", delim,
		encap_prefix, cmd, encap_infix, sno, encap_suffix,
		i < 2 ? delim_n : range,
		encap_prefix, cmd, encap_infix, eno, encap_suffix);
    }

    *saved = realloc(*saved, 1 + strlen(s) + strlen(*saved));
    strcat(*saved, s);
}

static void
merge_entries(entry_t *start, entry_t *end)
/* merge entries [start,..,end) into start */
{
    byte *saved = strdup("");
    byte *sno = NULL, *eno = NULL, *cmd = NULL;
    entry_t *e, *e1;

#ifdef DEBUG
    MESG((stderr, "[CCTmkind] Adding range: %d entries\n", end - start));
    dump_entry(start);
    dump_entry(end - 1);
#endif

    for (e = start; e < end; e++) {
	e1 = e;
	if (e->tag == MATCHED) continue;
	if (e->tag == REND) {
	    MESG((stderr, "[CCTmkind] Warning: unmatched end_range marker.\n"));
	}
	else if (e->tag == RSTART) {
	    while (++e1 < end && e1->tag != REND);
	    if (e1 >= end) {
		MESG((stderr, "[CCTmkind] Warning: unmatched start_range marker.\n"));
		e1 = e;
	    }
	    else {
		e1->tag = MATCHED;
	    }
	}
	e->tag = MATCHED;

	/* compare [e->pageno, e1->pageno] with current range */
	if (compare_strings(cmd, e->cmd, True) ||
	    compare_numbers(e->pageno, eno) > 1) {
	    /* flush current range */
	    if (sno != NULL) append_range(&saved, sno, eno, cmd, e);
	    sno = e->pageno;
	    eno = e1->pageno;
	    cmd = e->cmd;
	    continue;
	}
	if (compare_numbers(eno, e1->pageno) < 0) eno = e1->pageno;
    }
    if (sno != NULL) append_range(&saved, sno, eno, cmd, e - 1);

    if (start->cmd != NULL) {
	free(start->cmd);
	start->cmd = NULL;
    }
    free(start->pageno);
    start->pageno = saved;
    start->tag = NONE;
}

static int
merge_ind(void)
/* merge all identical entries. returns number of different entries */
{
    int i, j, count = 0;
    entry_t *e, *e1;

    i = 0;
    e = entries;
    while (i < entries_n) {
	if (e->tag == MATCHED) {
	    i++;
	    e++;
	    continue;
	}
	for (j = i + 1, e1 = e + 1; j  < entries_n; j++, e1++) { 
	    if (e1->tag == MATCHED) continue;
	    if (compare_entries(e, e1, True)) break;
	}
	merge_entries(e, e1);
	count++;
	e = e1;
	i = j;
    }

    return count;
}

static void
get_initial(byte *buffer, size_t buffersize, const byte *str)
{
    if (buffersize <= 0) return;

    if (*str >= 128) {
	GBK_get_initial(buffer, buffersize, str);
	buffer[0] = toupper(buffer[0]);
	return;
    }

    if (isalpha(*str)) {
	buffer[0] = (headings_flag >= 0 ? toupper : tolower)(*str);
        if (buffersize > 1) buffer[1] = '\0';
    }
    else if (is_digital_number(str) || roman_number(str)) {
        strncpy(buffer,
		headings_flag >= 0 ? numhead_positive : numhead_negative,
		buffersize);
	buffer[buffersize - 1] = '\0';
    }
    else {
        strncpy(buffer,
		headings_flag >= 0 ? symhead_positive : symhead_negative,
		buffersize);
	buffer[buffersize - 1] = '\0';
    }

    return;
}

static void
write_ind(FILE *f_out)
{
    char A[1024], B[sizeof(A) - 1];
    int i;
    entry_t *e, *e1 = &empty_entry;
    Boolean first_entry = True;
    int last_level = -1;
    int flag;		/* True ==> has printed for this entry */

    fprintf(f_out, "%s", preamble);

    if (start_no != NULL)
	fprintf(f_out, "%s%s%s", setpage_prefix, start_no, setpage_suffix);

    A[0] = '\0';
    B[sizeof(B) - 1] = '\0';
    for (i = 0, e = entries; i < entries_n; i++, e++) {
	if (e->tag == MATCHED) continue;
	get_initial(B, sizeof(B), e->key);
	if (strcmp(A, B)) {
	    strcpy(A, B);
	    if (!first_entry) fprintf(f_out, "%s", group_skip);
	    if (*A != '\0' && headings_flag) {
		fprintf(f_out, "%s%s%s", heading_prefix, A, heading_suffix);
	    }
	}
	first_entry = False;

	flag = False;
	if (compare_strings(e->item, e1->item, True)) {
	    fprintf(f_out, "%s%s", item_0, e->item);
	    last_level = 0;
	    flag = True;
	}
	if (e->subitem != NULL) {
	    if (compare_strings(e->subitem, e1->subitem, True) || 
		compare_strings(e->item, e1->item, True)) {
		fprintf(f_out, "%s%s",
			last_level == 0 ? (flag ? item_x1 : item_01) : item_1,
			e->subitem);
		last_level = 1;
		flag = True;
	    }
	    if (e->subsubitem != NULL) {
		fprintf(f_out, "%s%s", 
			last_level == 1 ? (flag ? item_x2 : item_12) : item_2,
			e->subsubitem);
		last_level = 2;
	    }
	}
	fprintf(f_out, "%s%s", e->pageno, delim_t);
	e1 = e;
    }

    fprintf(f_out, "%s", postamble);
}

int
main(int argc, char *argv[])
{
    int i;
    FILE *f_in = NULL, *f_out = NULL;
    char *output = NULL, *input = NULL;

    char **input_files = NULL;
    size_t input_files_n = 0;
    size_t input_files_max = 0;

    ist_init();
  
    for (i = 1; i < argc; i++) {
	if (argv[i][0] != '-') {
	    if (input_files_n >= input_files_max) {
		input_files = realloc(input_files,
				(input_files_max+=16) * sizeof(*input_files));
	    }
	    input_files[input_files_n++] = argv[i];
	    continue;
	}
	if (argv[i][2] != '\0') {
    error:
	   ERROR((stderr, "Invalid option (or argument for) \"%s\".\n",*argv));
    usage:
	   ERROR((stderr, "Usage:\n  cctmkind [-ilqrcgLT] [-s sty] [-o ind] "
			   "[-t log] [-p num] [-C sort] [idx ...]\n"));
	   ERROR((stderr, "Makeindex options:\n"));
	   ERROR((stderr, "  -c		removes leading/trailing blancs.\n"));
	   ERROR((stderr, "  -o ind	specifies output filename.\n"));
	   ERROR((stderr, "  -l		selects letter ordering.\n"));
	   ERROR((stderr, "  -p num	defines starting page number.\n"));
	   ERROR((stderr, "  -q		suppresses verbose messages.\n"));
	   ERROR((stderr, "Other options:\n"));
	   ERROR((stderr, "  -C sort	ordering ('pinyin' or 'stroke').\n"));
	   ERROR((stderr, "  -h		shows this help message.\n"));
	   ERROR((stderr, "Options '-irgLT' and '-t log' are ignored.\n"));
	   return 1;
	}
	switch (argv[i][1]) {
	    /* options with no arg */
	    case 'h': case 'H':
		goto usage;
	    case 'c':
		remove_blancs = True;
		break;
	    case 'l': 
		letter_ordering = True;
		break;
	    case 'q':
		quiet = True;
		break;
	    case 'i': case 'r': case 'g': 
	    case 'L': case 'T':
		MESG((stderr, "[CCTmkind] Warning: option \"%s\" ignored.\n",
					argv[i]));
		break;	/* ignored */
	
	    /* options with one arg */
	    case 's':
		ist_read(argv[++i]);
		break;
	    case 'o':
		output = strdup(argv[++i]);
		break;
	    case 'C':
		if (!stricmp(argv[i+1], "pinyin"))
		    GBK_set_ordering(GBK_PINYIN);
		else if (!stricmp(argv[i+1], "stroke"))
		    GBK_set_ordering(GBK_STROKE);
		else goto error;
		i++;
		break;
	    case 'p':
		start_no = argv[++i];	/* TODO: handle 'any', 'odd', 'even' */
		break;
	    case 't':
		MESG((stderr, "[CCTmkind] Warning: option \"%s %s\" ignored.\n",
					argv[i], argv[i + 1]));
		i++;
		break;	/* ignored */
		
	    default:
		goto error;
	}
    }

    MESG((stderr, "CCTmkind, v%s, by ZLB.\n", version));
    MESG((stderr, "Type \"cctmkind -h\" for help.\n"));

    if (input_files_n > 0 && output == NULL) {
	output = strdup(input_files[0]);
	i = strlen(output);
	while (i-- > 0 && output[i] != '.' && output[i] != '/'
#ifndef UNIX
			&& output[i] != '\\' && output[i] != ':'
#endif
	);
	if (i>=0 && output[i] == '.') output[i] = '\0';
	output = realloc(output, strlen(output) + 4 + 1);
	strcat(output, ".ind");
    }

    for (i = 0; i < (input_files_n ? : 1); i++) {
	if (input_files_n == 0) {
	    MESG((stderr,"[CCTmkind] Reading [stdin]..."));
	    f_in = stdin;
	    input = NULL;
	} else {
	    input = strdup(input_files[i]);
	    if ((f_in = fopen(default_ext(&input, ".idx"), "rt")) == NULL) {
		ERROR((stderr, "[CCTmkind] Error: cannot open input file \"%s\".\n",
					input));
		continue;
	    }
	    MESG((stderr,"[CCTmkind] Reading file \"%s\"...", input));
	}
	read_idx(f_in);
	if (f_in != stdin) fclose(f_in);
	if (input != NULL) free(input);
    }

    MESG((stderr, "[CCTmkind] Total number of entries: %d\n", (int)entries_n));

    default_ext(&output, ".ind");
    if (entries_n == 0) {
	if (output != NULL) {
	    MESG((stderr, "[CCTmkind] No output, deleting file \"%s\".\n", output));
	    unlink(output);
	} else {
	    MESG((stderr, "[CCTmkind] No output.\n"));
	}
	return 0;
    }

#ifdef DEBUG
    MESG((stderr, "[CCTmkind] ================= Before sorting\n"));
    dump_all_entries();
#endif
    
    /* sorting */
    qsort(entries, entries_n, sizeof(entry_t), sort_compare);

#ifdef DEBUG
    MESG((stderr, "[CCTmkind] ================= After sorting\n"));
    dump_all_entries();
#endif

    /* output results */
    if (output == NULL) {
	f_out = stdout;
    } else {
	if ((f_out = fopen(output, "w+t")) == NULL) {
	    ERROR((stderr, "[CCTmkind] Error: cannot open output file \"%s\".\n",
				    output));
	    return 1;
	}
    }
    merge_ind();	/* merge identical entries */
#ifdef DEBUG
    MESG((stderr, "[CCTmkind] ================= After merging\n"));
    dump_all_entries();
#endif
    write_ind(f_out);
    if (f_out != stdout) fclose(f_out); else fflush(f_out);
    if (output == NULL) {
	MESG((stderr, "[CCTmkind] Output written to [stdout].\n"));
    } else {
	MESG((stderr, "[CCTmkind] Output written to file \"%s\".\n", output));
    }

    return 0;
}
