/* $Id: cctmkind-ist.c,v 1.2 2004/11/26 11:41:32 zlb Exp $ */

#if defined(__MINGW32__) || defined(WIN32)
# define MIKTEX
#else
# undef MIKTEX
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#if defined(MIKTEX)
# include "windows.h"
  static HINSTANCE hClcltr = NULL;
  static char *pathlist = NULL;
  static int STDCALL (*miktex_initialize) (void);
  static int STDCALL (*miktex_uninitialize) (void);
  static int STDCALL (*miktex_find_file) (
	const char *FileName, const char *PathList, char *buffer);
  static char STDCALL *(*miktex_get_search_path) (
	const char *KeyList, const char *ValueName,
	char *PathList, DWORD MaxLen, const char *DefaultValue);
  static void FreeDllLibrary(void) {	/* called at exit to free DLLs */
	if (hClcltr != NULL) {
	    miktex_uninitialize();
	    FreeLibrary(hClcltr);
	    hClcltr = NULL;
	    if (pathlist != NULL) {free(pathlist); pathlist = NULL;}
	}
  }
#else
typedef unsigned char byte;	/* it is defined by windows.h */
#endif

#if defined (USE_KPSE)
#  include <kpathsea/c-auto.h>
#  include <kpathsea/kpathsea.h>
#endif

#if defined(WIN32) && !defined(__MINGW32__)
#  define PATH_MAX    _MAX_PATH+1
#else
#  include <limits.h>   /* PATH_MAX */
#endif

typedef enum {False, True} Boolean;

static void do_include(void);

#define CHAR_T   byte
#define STRING_T byte *
#define NUMBER_T int
#define XX1(spec, flag, var, type, func, arg) type var = (type)0;
#define AA1(spec, flag, var, type, func, arg) 
#define XX2(spec, flag, var, type, func, arg1, arg2) type var[] = {NULL, NULL};
#include "cctmkind-specs.h"

typedef enum {CHAR_T, NUMBER_T, STRING_T} spec_type_t;
typedef struct {
    char	*specifier;		/* specifier */
    Boolean	implemented;		/* False ==> not implemented */
    int		nargs;			/* number of arguments */
    void	*values;		/* pointer to attribute values */
    spec_type_t	type;			/* attribute type */
    void	(*func)(void);		/* specifier function */
} specifier_t;

specifier_t specifiers[] = {
#define XX1(spec,flag,var,type,func,arg) {spec, flag, 1, &var, type, func},
#define AA1(spec,flag,var,type,func,arg) {spec, flag, 1, &var, type, func},
#define XX2(spec,flag,var,type,func,arg1,arg2) {spec, flag, 2, var, type, func},
#include "cctmkind-specs.h"
};
static int specifiers_n = sizeof(specifiers)/sizeof(specifiers[0]);

static byte buffer[PATH_MAX < 4096 ? 4096 : PATH_MAX];

static int
spec_comp(const void *s1, const void *s2)
/* compare (the name of) two specifiers */
{
    return strcmp(((specifier_t *)s1)->specifier,
		  ((specifier_t *)s2)->specifier);
}

static byte *
read_token(byte *buf, FILE *f_ist)
/* reads an token, returns buf on success, NULL on EOF */
{
    int c, quote;
    byte *p = buf;

    do {
	if ((c = fgetc(f_ist)) == EOF) return NULL;
	if (c == '%') {
	    while ((c = fgetc(f_ist)) != '\n' && c != EOF);
	    if (c == EOF) return NULL;
	}
    } while (isspace(c));

    *(p++) = c;
    quote = (c == '\'' || c == '"') ? c : -1;
    while (1) {
	if ((c = fgetc(f_ist)) == EOF) break;
	if (quote == -1 && c == '%') {
	    while ((c = fgetc(f_ist)) != '\n' && c != EOF);
	    break;
	}
        if (quote == -1 && isspace(c)) {
	    ungetc(c, f_ist);
	    break;
	}
	if (c == '\\') {
	    if ((c = fgetc(f_ist)) == EOF) return NULL;
	    switch (c) {
		case 'n' : *(p++) = '\n'; break;
		case 't' : *(p++) = '\t'; break;
		default  : *(p++) = c; break;
	    }
	    continue;
	}
	*(p++) = c;
	if (quote != -1 && c == quote) break;
    }
    *p = '\0';
#if 0
    fprintf(stderr, "token=|%s|\n", buf);
#endif
    return buf;
}

static Boolean
get_value(byte *buf, spec_type_t type, void *values, int j)
/* scans the j-th value of a specifier, returns False on error */
{
    size_t l = (buf == NULL) ? 0 : strlen(buf) - 1;

    if (values == NULL) return True;

    switch (type) {
	case CHAR_T:
	    if (l != 2 || buf[0] != '\'' || buf[2] != '\'') return False;
	    *((byte *)values+j) = buf[1];
	    break;
	case STRING_T:
	    if (buf == NULL) {
		if (*((byte **)values+j) != NULL) free(*((byte **)values+j));
		*((byte **)values+j) = NULL;
		break;
	    }
	    /* Note: unqoted string is allowed because we treate numbers
	     * as strings for the specifiers group_head and sort_group */
	    if ((buf[0] == '"' && buf[l] == '"')) {
		buf[l] = '\0';
		buf++;
	    }
	    *((byte **)values+j) = realloc(*((byte **)values+j), strlen(buf)+1);
	    strcpy(*((byte **)values+j), buf);
	    break;
	case NUMBER_T:
	    if (sscanf(buf, "%d", (int *)values+j) != 1) return False;
	    break;
    }

    return True;
}

#ifdef TEST
static void
dump_specifier(specifier_t *sp)
/* print the values if the i-th specifier */
{
    int j;
    void *value;

#if 0
    return;
#endif
    
    if (sp == NULL) return;

    fprintf(stdout, "%s", sp->specifier);
    value = sp->values;
    for (j = 0; value != NULL && j < sp->nargs; j++) {
	switch (sp->type) {
	    case CHAR_T:
		fprintf(stdout, " '%c'", *((byte *)value + j));
		break;
	    case STRING_T:
		fprintf(stdout, " \"%s\"", *((byte **)value + j));
		break;
	    case NUMBER_T:
		fprintf(stdout, " %d", *((int *)value + j));
		break;
	}
    }
    fprintf(stdout, "\n");
}
#endif

void
ist_init(void)
/* Assign default values of specifiers */
{
    static Boolean initialized = False;

    if (initialized) return;
    initialized = True;
#define XX1(spec, flag, var, type, func, arg) get_value(arg, type, &var, 0);
#define AA1(spec, flag, var, type, func, arg)
#define XX2(spec, flag, var, type, func, arg1, arg2) \
    get_value(arg1, type, var, 0); get_value(arg1, type, var, 1);
#include "cctmkind-specs.h"
    /* Sorting the specifier table for allowing binary search */
    qsort(specifiers, specifiers_n, sizeof(specifiers[0]), spec_comp);

#ifdef TEST
    {
	int i;
	fprintf(stderr, "============================= sorted specifiers\n");
	for (i = 0; i < specifiers_n; i++) dump_specifier(specifiers + i);
	fprintf(stderr, "===============================================\n");
    }
#endif
}

#ifdef UNIX
#  define DIR_SEP '/'
#  define PATH_SEP ':'
#else
#  define DIR_SEP '\\'
#  define PATH_SEP ';'
#endif

#if !defined(USE_KPSE) && !defined(MIKTEX)
static byte *
find_file_in_1dir(byte *dir, const byte *name, FILE **f)
/* search file in a path, expanding '{..., ...}' components */
{
    static byte fname[PATH_MAX];
    static Boolean error;
    byte *p, *q, *r, *list, *remain;

#ifdef TEST
    fprintf(stderr, "Look at directory \"%s\"...\n", dir);
#endif

    if ((p = strchr(dir, '{')) == NULL) {
	size_t l = strlen(dir);

	if (l <= 0) return NULL;
	/* FIXME: check for buffer overflow with 'fname' */
	memcpy(fname, dir, l);
	if (fname[l - 1] != DIR_SEP) fname[l++] = DIR_SEP;
	strcpy(fname + l, name);
#ifdef TEST
	fprintf(stderr, "Trying file \"%s\"...\n", fname);
#endif
	return (*f = fopen(fname, "rt")) == NULL ? NULL : fname;
    }

    /* find closing '}' */
    if ((q = strchr(p + 1, '}')) == NULL) {
	fprintf(stderr, "[CCTmkind] Error: invalid path \"%s\".\n", dir);
	error = True;
	return NULL;
    }

    error = False;
    *q = '\0';
    list = strdup(p + 1);
    remain = strdup(q + 1);
    q = list;
    while (q != NULL && !error) {
	if ((r = strchr(q, ',')) != NULL) 
	    *(r++) = '\0';
	else
	    r = NULL;
	strcpy(p, q);
	strcat(p, remain);
	if ((q = find_file_in_1dir(dir, name, f)) != NULL) {
	    free(list);
	    free(remain);
	    return q;
	}
	q = r;
    }
    free(list);
    free(remain);

    return NULL;
}

static byte *
find_file_in_dirs(byte *dirs, const byte *name, FILE **f)
/* finds and opens the file named 'name' in the list of directories in 'dirs'.
 * returns the full filename or NULL in case of failure
 *
 * TODO: recursion into subdirs in the case of traling '//' (or '\\') */
{
    byte *p;

    /* change ';' to ':' (UNIX), '/' to '\\' (WIN32/DOS) in dirs */
    for (p = dirs; *p != '\0'; p++) {
#ifdef UNIX
	if (*p == ';') *p = PATH_SEP;
#else
	if (*p == '/') *p = DIR_SEP; 
#endif
    }

    while (dirs != NULL) {
	byte *q;
	while (isspace(*dirs)) dirs++;	/* skip leading spaces in a path */
	if ((q = strchr(p = dirs, PATH_SEP)) != NULL) {
	    *q = '\0';
	    dirs = q + 1;
	}
	else {
	    dirs = NULL;
	}
	if ((p = find_file_in_1dir(p, name, f)) != NULL) return p;
    }

    return NULL;
}
#endif	/* !defined(USE_KPSE) && !defined(MIKTEX) */

void
ist_read(const char *istfile)
{
#if defined(MIKTEX) || defined(USE_KPSE)
    static Boolean initialized = False;
#endif
    static specifier_t spec = {buffer, False, 0, NULL, STRING_T, NULL};
    specifier_t *sp;
    const byte *fn = NULL;
    FILE *f_ist;
    int j;

    ist_init();

    if ((f_ist = fopen(istfile, "rt")) == NULL) {
#if !defined(USE_KPSE) && !defined(MIKTEX)
	/* check for INDEXSTYLE environment variable */
	byte *env = getenv("INDEXSTYLE");
	if (env == NULL)
	    env = "C:\\CTEX\\TEXMF\\MAKEINDEX\\"
		  "{,LATEX,BABEL,BASE,ENGLISH,GERMAN,CCT}";
#ifdef TEST
	fprintf(stderr, "[CCTmkind] INDEXSTYLE: |%s|\n", env);
#endif
	if (env != NULL) {
	    env = strdup(env);
	    fn = find_file_in_dirs(env, istfile, &f_ist);
	    free(env);
	    env = NULL;
	}
#else	/* !defined(USE_KPSE) && !defined(MIKTEX) */
	/* search TEXMF */
#if defined(MIKTEX)
	if (!initialized) {
	    char *p;
	    initialized = True;
	    if ((hClcltr = LoadLibrary(p = "MiKTeX-core-0.dll")) == NULL &&
		(hClcltr = LoadLibrary(p = "MiKTeX-core-1.dll")) == NULL &&
		(hClcltr = LoadLibrary(p = "MiKTeX-core-2.dll")) == NULL &&
		(hClcltr = LoadLibrary(p = "MiKTeX-core-3.dll")) == NULL &&
		(hClcltr = LoadLibrary(p = "MiKTeX-core-4.dll")) == NULL) {
		fprintf(stderr, "Can't load MiKTeX-core DLL\n");
		goto error;
	    }
#ifdef TEST
	    fprintf(stderr, "[CCTmkind] Loaded DLL library %s.\n", p);
#endif
	    atexit(FreeDllLibrary);
#define FIND_FUNC(func, name)					\
    func = (void *)GetProcAddress(hClcltr, name);		\
    if (func == NULL) {						\
	fprintf(stderr, "[CCTmkind] Abort: no function " name " in DLL.\n"); \
	goto error;						\
    }
	    FIND_FUNC(miktex_initialize, "_miktex_initialize@0")
	    FIND_FUNC(miktex_uninitialize, "_miktex_uninitialize@0")
	    FIND_FUNC(miktex_find_file, "_miktex_find_file@12")
	    FIND_FUNC(miktex_get_search_path, "_miktex_get_search_path@20")
#undef FIND_FUNC
	    miktex_initialize();
	    pathlist =  miktex_get_search_path("Makeindex", "INDEXSTYLE",
				buffer, sizeof(buffer), ".");
	    if (pathlist != NULL) pathlist = strdup(buffer);
#ifdef TEST
	    fprintf(stderr, "[CCTmkind] Path List = %s\n", pathlist);
#endif
	}
	if (pathlist != NULL) {
#  ifdef TEST
	    fprintf(stderr, "[CCTmkind] Calling miktex_find_file ...\n");
#  endif
	    if (miktex_find_file(istfile, pathlist, buffer)) fn = buffer;
	}
#elif defined(USE_KPSE)	/* defined MIKTEX */
	if (!initialized) {
	    kpse_set_program_name("cctmkind", "cct");
	    initialized = True;
	}
#  ifdef TEST
	fprintf(stderr, "[CCTmkind] Calling kpse_find_file ...\n");
#  endif
	fn = kpse_find_file(istfile, kpse_ist_format, 1);
#endif	/* defined(MIKTEX) */

	if (fn != NULL) {
	    if ((f_ist = fopen(fn, "rt")) == NULL) fn = NULL;
	}
#endif	/* !defined(USE_KPSE) && !defined(MIKTEX) */

	if (fn == NULL) {
#ifdef MIKTEX
    error:
#endif
	    fprintf(stderr, "[CCTmkind] Error: can't open index file "
			    "\"%s\".\n", istfile);
	    return;
	}
    }
    else {
	fn = istfile;
    }
    fprintf(stderr, "[CCTmkind] Loading index style \"%s\".\n", fn);

    while (1) {
	if (read_token(buffer, f_ist) == NULL) break;
	if (!isalpha(buffer[0])) continue; /* attribute, silently ignored */
	if ((sp = bsearch(&spec, specifiers, specifiers_n,
			   sizeof(spec), spec_comp)) == NULL) {
	    fprintf(stderr, "[CCTmkind] Error: invalid specifier \"%s\", "
			    "ignored.\n", buffer);
	    continue;
	}
	for (j = 0; j < sp->nargs; j++) {
	   if (read_token(buffer, f_ist) == NULL || !get_value(buffer,
		sp->type, sp->values, j)) {
		fprintf(stderr, "[CCTmkind] Error: invalid value for specifier "
				"\"%s\", ignored.\n", sp->specifier);
		continue;
	   }
	}
#ifdef TEST
	dump_specifier(sp);
#endif
	if (!sp->implemented) {
	    fprintf(stderr, "[CCTmkind] Warning: specifier \"%s\" is not "
			    "implemented.\n", sp->specifier);
	    sp->implemented = True;	/* this avoids multiple warnings */
	    continue;
	}
	if (sp->func != NULL) sp->func();
    }
    fclose(f_ist);

    return;
}

/*--------------------------- Specifier functions -----------------------*/
static void
do_include(void)
/* processing include file */
{
    byte *p;

    ist_read(p = include);
    free(p);
    include = NULL;
}

/*-----------------------------------------------------------------------*/

#ifdef TEST
int 
main(int argc, char *argv[])
{
    int i;

    if (argc != 2) {
	fprintf(stderr, "Usage: %s filename.ist\n", argv[0]);
	return 1;
    }

    ist_read(argv[1]);

    fprintf(stderr, "======================================================\n");
    for (i = 0; i < specifiers_n; i++) dump_specifier(specifiers + i);
    
    return 0;
}
#endif
