/* $Id: patchchk.c,v 1.2 2004/11/26 11:29:07 zlb Exp $ */

typedef unsigned char byte;

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include "common.h"

#define dvi_nop		138
#define dvi_fnt_def1	243
#define dvi_post	248
#define dvi_post_post	249

#define factor	((double)num/(double)den*1e-5/2.54*72.27*mag*0.001)

static FILE *f_dvi = NULL;

static long num, den, mag=0l;

static long 
readNumber(int size)
/* read a number of size bytes from f_dvi. Since the byte ordering in
   a DVI file is MSB first, we have to reverse the bytes read */
{
  long ll = 0;

  while (size-- > 0)
    ll = 256L * ll + (unsigned char )(fgetc(f_dvi));

  return ll;
}

int 
patchdvi_check(char *dvi_name)
/* returns TRUE if the dvi file 'dvi_name' is in the CCT format,
   i.e., it needs to be dvipatched. */
{
  int i, b;
  long fpos, ll;
  char s[1024];

  assert ( sizeof(long) >= 4 );

  if (f_dvi != NULL) {
    fclose(f_dvi);
    f_dvi=NULL;
  }

  /* append .dvi to file name when necessary */
  if (*(get_file_ext(dvi_name)) == '\0') {
    strcpy(s, dvi_name);
    strcat(s, ".dvi");
    dvi_name = s;
  }

  if ((f_dvi=fopen(dvi_name,"rb"))==NULL) {
    fprintf(stderr,"patchdvi_check: cannot open DVI file \"%s\"!\n", dvi_name);
    return 0;
  }
  setvbuf(f_dvi, NULL, _IOFBF, 32*1024);

  fseek(f_dvi, -4l, SEEK_END);
  fpos=ftell(f_dvi);
  ll = readNumber(4);
  if (ll != 0xdfdfdfdfl) {
dvi_error:
    fprintf(stderr,"patchdvi_check: not a valid dvi file!\n");
abort:
    fclose(f_dvi);
    f_dvi=NULL;
    return 0;
  }

  /* skip trailing 0xdf's */
  do {
    fseek(f_dvi,--fpos,SEEK_SET);
    i=fgetc(f_dvi);
  } while (i==0xdf);

  /* check DVI file id. */
  if (i!=2) goto dvi_error;

  /* read "post_post" command */
  fseek(f_dvi,fpos-5l,SEEK_SET);
  if (fgetc(f_dvi)!=dvi_post_post) {
    fprintf(stderr,"patchdvi_check: \"post_post\" command not found!\n");
    goto abort;
  }

  /* goto postamble, check for the "post" command byte, and read
     some important parameters from the postamble */
  fpos = readNumber(4);
  fseek(f_dvi, fpos, SEEK_SET);
  if (fgetc(f_dvi)!=dvi_post) {
    fprintf(stderr,"patchdvi_check: \"post\" command expected here!\n");
    goto abort;
  }

  fseek(f_dvi, 4l, SEEK_CUR);		/* skip location of the last page */
  num = readNumber(4);			/* read numerator */
  den = readNumber(4);			/* read denominator */
  if (!mag) mag = readNumber(4);		/* read magnification */

  /* process font definitions */
  fseek(f_dvi,fpos+29l,SEEK_SET);
  do {
    long ssize, dsize, chksum;

    if ((b=fgetc(f_dvi))==dvi_nop) continue;
    if (b<dvi_fnt_def1 || b>=dvi_fnt_def1+4) {
      if (b!=dvi_post_post) {
	fprintf(stderr,"patchdvi_check: forbidden command in the postamble\n");
	goto abort;
      }
      break;
    }
    /* read font size and font checksum */
    ll = readNumber(b-dvi_fnt_def1+1);	/* read font no. */
    chksum = readNumber(4);
    ssize = readNumber(4);
    dsize = readNumber(4);

    /* read the font name */
    i=fgetc(f_dvi); i+=fgetc(f_dvi);	/* i=length of font name string */
    fread(s,i,1,f_dvi); s[i]='\0';

#ifdef DEBUG
    fprintf(stderr, "fontno=%-4ld  design size=%-6.4lg  scale size=%-6.4lg  "
		    "name=%s\n", ll, factor*dsize, factor*ssize, s);
#endif

    /* check for CCT's special font names */
    if (!strcmp("ccdummy",s) || !strncmp("ccfnt",s,5)) {
      fclose(f_dvi);
      f_dvi=NULL;
      return 1;
    }
  } while (1);

  fclose(f_dvi);
  f_dvi=NULL;

  return 0;
}

#ifdef TEST
char *get_file_ext(char *fn)
{
    size_t l=strlen(fn);
    char *p=fn+l-1;

#ifdef UNIX
    while (p >= fn && *p != '.' && *p != '/')
#else
    while (p >= fn && *p != '.' && *p != '/' && *p!='\\' && *p != ':')
#endif
        p--;
    if (p<fn || *p!='.') p=fn+l;

    return p;
}

int main(int argc, char *argv[])
/* This program returns 0 if dvi file is CCT format, 1 otherwise */
{
    char *dvi_name;

    if (argc != 2) {
	fprintf(stderr, "Usage: %s dvi_name\n", argv[0]);
	return 0;
    }
    dvi_name = argv[1];
    
    return !patchdvi_check(dvi_name);
}
#endif
