/* $Id: ctex.c,v 1.5 2005/06/28 13:59:17 zlb Exp $
 *
 * An universal frontend for processing CCT/CJK Chinese TeX documents.
 *
 *
 * Author: Zhang Linbo <zlb@lsec.cc.ac.cn>.
 *
 * WIN32 port (MS Visual C) by nsii and hooklee.
 *
 * The processing is divided into three stages:
 *
 *   Stage 1: Preprocessing
 * 	cct or cctspace, cctconv
 *
 *   Stage 2: TeXing
 * 	tex, amstex, pdftex, latex, pdflatex
 *
 *   Stage 3: Creating PS/PDF output
 * 	dvips, dvipdfmx
 */

#if !defined(UNIX) && !defined(MIKTEX)
#  define MIKTEX
#endif

#ifndef USE_GLOB
#  define USE_GLOB 1
#endif

#ifndef TEX_OPTS
#  define TEX_OPTS "--translate-file=cp8bit.tcx"
#endif

#ifndef CP_CMD
#  define CP_CMD "/bin/cp -f"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <limits.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#ifdef WIN32
#  include <direct.h>
#  include <io.h>
#  include <process.h>
#else
#  include <unistd.h>
#  if USE_GLOB
#    include <glob.h>
#  endif
#endif
#ifdef GO32
#  include <dir.h>
#endif

#include "common.h"

/* Value to use if corresponding environment variable is not set */
#ifdef UNIX
#  define CCPKPATH "/usr/share/texmf/fonts/cct/pk"
#  define CCHZPATH "/usr/share/texmf/fonts/cct/hzfonts"
#else
#  define CCPKPATH "C:\\CTeX\\localtexmf\\fonts\\pk\\modeless\\cct\\dpi$d"
#  define CCHZPATH "C:\\CTeX\\localtexmf\\cct\\fonts"
#endif

static char *SEP = "=========================================================";

/* tags for Stage 1, CCT: cct, CCTCONV: cctconv, CCTSPACE: cctspace+cctconv */
enum {NONE, CCT, CCTCONV, CCTSPACE};

/* tags for Stage 2 */
enum
    {NONE1,  TeX,   AmSTeX,   PDFTeX,   LaTeX,   AmSLaTeX,   PDFLaTeX};
static char *stage2_cmds[] =
    {"true", "tex", "amstex", "pdftex", "latex", "amslatex", "pdflatex", NULL};

/* tags for Stage 3 */
enum {NONE2, DVIPS, DVIPDFMX};

/* User-customizable parameters. */
static int stage1 	= CCTSPACE;
static int stage2	= LaTeX;
static int stage3	= NONE;

static int dpi		= 600;		/* DPI (for patchdvi, dvips, etc.) */
static int max_depth	= 8;		/* maxi. depth of nested \input's */
static int latex_runs	= 4;		/* maxi. number of times to run latex */
static Boolean ctx_flag	= False;	/* whether to run cct.exe for*/
static Boolean verbose	= True;		/* whether to show messages */
static Boolean cctspace	= True;		/* whether to run `cctspace' */
static Boolean force_pre= False;	/* don't compare timestamps */
static Boolean force_tilde= False;	/* force cctspace to insert '~' */
static Boolean clean	= False;	/* clean temp dir */
static Boolean help	= False;	/* display help message */
#ifdef UNIX
static Boolean patchdvi	= False;
#else
static Boolean patchdvi = True;
#endif
static Boolean bibtex	= False;
#ifdef USE_KPSE
       Boolean use_kpse	= False;	/* exported to ctexscan.c */
#endif
static Boolean xref	= False;

/* Warning: the string variables below must be initialized in main() */
static char *gbk2uni_opts = NULL;
static char *bibtex_opts = NULL;
static char *makeindex_opts = NULL;
static char *glossary_opts = NULL;
static char *makeindex_prog = NULL;
static char *dvips_opts = NULL;
static char *dvipdfmx_opts = NULL;
#ifndef UNIX
  static char *patchdvi_opts = NULL;
#endif

static FILE *f_xref	= NULL;

/* Variables */
static int depth = 0;			/* current depth of nested \input's */
static char buffer[16*1024];		/* global buffer */
static char main_file[PATH_MAX];	/* (converted) main filename */
static char main_base[PATH_MAX];	/* main filename without extension */
static char *temp_file = NULL;
static struct stat buf, buf1;		/* for file status or timestamp */

/* data structure for cmdline options */
typedef enum {VAR_NONE, VAR_INT, VAR_BOOLEAN, VAR_STRING} VarType;
typedef struct {
    char	*name;		/* option name without leading dash */
    char	*help;		/* help text for this option */
    void 	*var;		/* address of the variable to assign value to */
    VarType	var_type;	/* type of the variable */
    Boolean	has_arg;	/* True if option has the "=value" part */
    int		value;		/* value to assign to the variable if
				 * 'has_arg' is False. If 'has_arg' is True,
				 * then this field is not used and the value
				 * is obtained from the '=VALUE' part
				 * in the option string. */
} OPTION;

/* cmdline options */
#define NOPTIONS	(sizeof(options)/sizeof(options[0]))
OPTION options[] = {
    {"dpi",		"resolution (DPI)",
	&dpi,		VAR_INT,	True,	600},
    {"max-depth",	"maxi. depth of nested \\input's",
	&max_depth,	VAR_INT, 	True,	8},
    {"latex-runs",	"maxi. number of times to run latex",
	&latex_runs,	VAR_INT,	True,	4},
    {"quiet",		"run quietly",
	&verbose,	VAR_BOOLEAN,	False,	False},
    {"no-cctspace",	"don't run `cctspace'",
	&cctspace,	VAR_BOOLEAN,	False,	False},
#ifndef UNIX
    {"no-patchdvi",	"don't run `patchdvi'",
	&patchdvi,	VAR_BOOLEAN,	False,	False},
#endif
    {"force-tilde",	"force cctspace to insert '~' instead of ' '",
	&force_tilde,	VAR_BOOLEAN,	False,	True},

    {"tex",		"process the document with `tex'",
	&stage2,	VAR_INT,	False,	TeX},
    {"amstex",		"process the document with `amstex'",
	&stage2,	VAR_INT,	False,	AmSTeX},
    {"pdftex",		"process the document with `pdftex'",
	&stage2,	VAR_INT,	False,	PDFTeX},
    {"latex",		"process the document with `latex' (default)",
	&stage2,	VAR_INT,	False,	LaTeX},
    {"amslatex",	"process the document with `amslatex'",
	&stage2,	VAR_INT,	False,	AmSLaTeX},
    {"pdflatex",	"process the document with `pdflatex'",
	&stage2,	VAR_INT,	False,	PDFLaTeX},

    {"dvips",		"run `dvips' after TeXing the document",
	&stage3,	VAR_INT,	False,	DVIPS},
    {"dvipdfmx",	"run `dvipdfmx' after TeXing the document",
	&stage3,	VAR_INT,	False,	DVIPDFMX},
    {"bibtex",		"run `bibtex' to generate .bbl file",
	&bibtex,	VAR_BOOLEAN,	False,	True},

    {"force-pre",	"force (re)preprocessing of all input files",
	&force_pre,	VAR_BOOLEAN,	False,	True},

    /* options to pass to postprocessors */
    {"gbk2uni_opts",	"options to pass to `gbk2uni'",
	&gbk2uni_opts,	VAR_STRING,	True,	0},
    {"bibtex_opts",	"options to pass to `bibtex'",
	&bibtex_opts,	VAR_STRING,	True,	0},
    {"makeindex_prog",	"makeindex program",
	&makeindex_prog,	VAR_STRING,	True,	0},
    {"makeindex_opts",	"makeindex options",
	&makeindex_opts,VAR_STRING,	True,	0},
    {"glossary_opts",	"options for glossaries",
	&glossary_opts,VAR_STRING,	True,	0},
    {"dvips_opts",	"options to pass to `dvips'",
	&dvips_opts,VAR_STRING,	True,	0},
    {"dvipdfmx_opts",	"options to pass to `dvipdfmx'",
	&dvipdfmx_opts,VAR_STRING,	True,	0},
#ifndef UNIX
    {"patchdvi_opts",	"options to pass to `patchdvi'",
	&patchdvi_opts,VAR_STRING,	True,	0},
#endif

#ifdef USE_KPSE
    {"kpse",		"use Kpathsea library call to find input files",
	&use_kpse,	VAR_BOOLEAN,	False,	True},
#endif
    {"xref",		"generate .xref file",
	&xref,		VAR_BOOLEAN,	False,	True},

    {"clean",		"remove `"CTEX_TMP"*' (ignoring other arguments)",
	&clean,		VAR_BOOLEAN,	False,	True},
    {"help",		"display this help message",
	&help,		VAR_BOOLEAN,	False,	True}
};

static fnmap_t *fnmap = NULL;
static size_t fnmap_n = 0, fnmap_max = 0;

static void
add_to_fnmap(const char *orig_fn, const char *temp_fn)
{
    if (fnmap_n >= fnmap_max) {
	fnmap = realloc(fnmap, (fnmap_max += 16) * sizeof(fnmap_t));
	if (fnmap == NULL) {
	    fprintf(stderr, "add_to_fnmap: Cannot allocate memory!\n");
	    exit(1);
	}
    }
    fnmap[fnmap_n].orig_fn = strdup(orig_fn);
    /* strip the ctextemp_ prefix to save memory */
    fnmap[fnmap_n].temp_fn = strdup(temp_fn + sizeof(CTEX_TMP) - 1);
    fnmap_n++;
}

static void
message(char *s)
{
    int i;
    char msg[80];

    if (!verbose) return;

    i = /* depth * 2 + 3*/ 3;
    memset(msg, '+', i);
    msg[i++] = ' ';
    if (i + strlen(s) < 80)
	strcpy(msg + i, s);
    else {
	strncpy(msg + i, s, 75 - i);
	strcpy(msg + 75, " ...");
    }
    fprintf(stderr, "%s\n", msg);
}

#ifdef WIN32
/* arguments list for _spawnvp */
static char **args = NULL;
size_t args_max = 0;

static char *
get_next_arg(char **cmd)
{
    char *p;
	
    while (isspace(**cmd))
	(*cmd)++;

    if (**cmd == '\0')
	return NULL;

    /* TODO: quotes within an argument, e.g., file" name", and escape chars */
 
    if (**cmd == '"' || **cmd == '\'') {
	for (p = *cmd + 1; *p != '\0' && *p != **cmd; p++);
	if (*p == **cmd)
	    p++;
    }
    else {
	for (p = *cmd; *p != '\0' && !isspace(*p); p++);
    }

    return p;
}
#endif

static int
do_cmd(char *cmd, Boolean abort_on_error)
{
    int ret;

#ifndef WIN32
    message(cmd);
    ret = system(cmd);
#else
    int nargs = 0;
    char *p, *q;

    message(q = cmd);
    while (True) {
	p = get_next_arg(&q);
	if (p == NULL)
	    break;
	if (nargs + 1 >= args_max) {
	    if ((args = realloc(args, (args_max+=16)*sizeof(*args))) == NULL) {
	mem_err:
		fprintf(stderr, "Memory allocation error, abort.\n");
		exit(1);
	    }
	}
	if ((args[nargs] = malloc(p - q + 1)) == NULL)
	    goto mem_err;
	memcpy(args[nargs], q, p - q);
	args[nargs][p - q] = '\0';
	nargs++;
	q = p;
    }
    if (nargs > 0) {
	/* check if the command pathname has an extension part */
	p = args[0] + strlen(args[0]) - 1;
	while (p > args[0] && *p != '/' && *p != '\\' && *p != '.')
	    p--;
	if (*p != '.') {
	    /* append .exe if no extension is specified */
	    if ((args[0] = realloc(args[0], strlen(args[0]) + 1 + 4)) == NULL)
		goto mem_err;
	    strcat(args[0], ".exe");
	}
	args[nargs] = NULL;
	ret = _spawnvp(P_WAIT, args[0], (void *)args);
	/* free memory */
	while (--nargs >= 0)
	    free(args[nargs]);
    }
    else
	ret = 0;
#endif
    if (ret && abort_on_error) {
	fprintf(stderr, "Cannot execute \"%s\", abort.\n", cmd);
	exit(ret);
    }
    return ret;
}

static void
rename_file(const char *oldname, char *newname)
{
    static char buf[2*PATH_MAX + 128];
#ifdef UNIX
    /* UNIX: use '/bin/cp -f' to make sure that the new file
       has the same inode. TODO: shall we also do it for Windows? */
    sprintf(buf, CP_CMD " \"%s\" \"%s\"", oldname, newname);
    do_cmd(buf, True);
    unlink(oldname);
#else
    sprintf(buf, "move \"%s\" -> \"%s\"", oldname, newname);
    message(buf);
    unlink(newname);
    if (rename(oldname, newname)) {
	fprintf(stderr, "Cannot move \"%s\" -> \"%s\", abort.\n",
			oldname, newname);
	exit(1);
    }
#endif
}

static void
append_file(char **list, size_t *size, char *fn)
{
    size_t len = strlen(fn) + 1;

    *list = realloc(*list, (*size) + len);
    if (*list == NULL) {
	fprintf(stderr, "Cannot allocate memory, abort.\n");
	exit(1);
    }
    memcpy((*list) + (*size), fn, len);
    (*size) += len;
}

static void loop(char *fn);

static void
process_file(const char *fn, const char *fnout, Boolean copy_flag)
/* Preprocess 'fin'.
 * copy_flag == True ==> copy the file to work dir but don't process it. */
{
    static char lstfile[PATH_MAX], outfile[PATH_MAX];

    /* check modification time of the input (original) file */
    if (stat(fn, &buf)) {
	fprintf(stderr, "Can't find %s.\nThis shouldn't happen "
			"(try 'ctex -clean' first).\n", fn);
	exit(1);
    }

    /* make output filename */
#ifdef USE_TEMPDIR
    sprintf(outfile, CTEX_TMP "%s", fnout);
#else
    strcpy(outfile, fnout);
#endif
    
#ifdef DEBUG
    fprintf(stderr, "process_file: depth=%d, fn=%s, outfile=%s\n",
		    depth, fn, outfile);
#endif

    add_to_fnmap(fn, outfile);

    if (!depth) {
	strcpy(main_file, fnout);
	strcpy(main_base, fnout);
	*(get_file_ext(main_base)) = '\0';
    }

#ifdef USE_TEMPDIR /* copy_flag is always False if USE_TEMPDIR undefined */
    if (copy_flag) { /* copy (graphics) file to tmp directory */
#ifndef UNIX
	FILE *src, *dst;
#endif
	/* compare timestamps */
	if (!stat(outfile, &buf1) && buf.st_mtime <= buf1.st_mtime
		&& !force_pre) return;
	unlink(outfile);
#ifdef UNIX
	/* Instead of copying file, create a symbolic link */
	if (*fn != '/') {	/* relative path, append "../" */
	    memcpy(lstfile, "../", 3);
	    strcpy(lstfile + 3, fn);
	    fn = lstfile;
	}
	sprintf(buffer, "ln -sf %s %s\n", fn, outfile);
	message(buffer);
	symlink(fn, outfile);
#else
	sprintf(buffer, "copy %s %s\n", fn, outfile);
	message(buffer);
	if ((src = fopen(fn, "rb")) == NULL) {
	    fprintf(stderr, "warning: can't copy graphics file \"%s\".\n", fn);
	    return;
	}
	setvbuf(src, NULL, _IOFBF, 32*1024);
	if ((dst = fopen(outfile, "w+b")) == NULL) {
	    fprintf(stderr, "warning: can't create file \"%s\".\n", outfile);
	    fclose(src);
	    return;
	}
	setvbuf(dst, NULL, _IOFBF, 32*1024);
	while (1) {
	    size_t len, trunc = sizeof(buffer);
	    len = fread(buffer, 1, trunc, src);
	    if (len) fwrite(buffer, 1, len, dst);
	    if (len < trunc) break;
	}
	fclose(src);
	fclose(dst);
#endif
	return;
    }
#endif	/* USE_TEMPDIR */

    /* make list filename */
    sprintf(lstfile, "%s.lst", outfile);

    /* compare timestamps */
    if (stat(outfile, &buf1) || buf.st_mtime > buf1.st_mtime
		|| force_pre || buf1.st_size <= 1 /* empty file */) {
#ifdef UNIX 
	FILE *pipe;
#endif
	Boolean local_cctspace = cctspace;

	/* check the first line for the '%& -no-cctspace' option */
	if (local_cctspace) {
	    FILE *src;
	    if ((src = fopen(fn, "rt")) != NULL) {
		if (fgets(buffer, sizeof(buffer), src) != NULL &&
		    memcmp(buffer, "%&", 2) == 0 && 
		    strstr(buffer, "-no-cctspace") != NULL)
		    local_cctspace = False;
		fclose(src);
	    }
	}
#ifdef UNIX 
	if (ctx_flag) {
	    /* run cct.exe */
	    sprintf(buffer, "ctexscan \"%s\" | cct -q - > \"%s\"", fn, outfile);
	    message(buffer);
	    sprintf(buffer, "cct -q - > \"%s\"", outfile);
	}
	else if (local_cctspace) {
	    /* run cctspace and cctconv */
	    sprintf(buffer, "ctexscan \"%s\" | cctspace %s | cctconv > \"%s\"",
			    fn, force_tilde ? "-t" : "", outfile);
	    message(buffer);
	    sprintf(buffer, "cctspace %s | cctconv > \"%s\"", 
			    force_tilde ? "-t" : "", outfile);
	}
	else {
	    /* run cctconv */
	    sprintf(buffer, "ctexscan \"%s\" | cctconv > \"%s\"", fn, outfile);
	    message(buffer);
	    sprintf(buffer, "cctconv > \"%s\"", outfile);
	}
	if ((pipe = popen(buffer, "w")) == NULL) {
	    fprintf(stderr, "Cannot execute '%s', abort.\n", buffer);
	    exit(1);
	}
	ctexscan(fn, lstfile, ctx_flag, pipe);
	pclose(pipe);
#else
	FILE *f;

	sprintf(buffer, "ctexscan: %s -> %s", fn, temp_file);
	message(buffer);
	if ((f = fopen(temp_file, "w+t")) == NULL) {
	    fprintf(stderr, "Cannot create '%s', abort.\n", temp_file);
	    exit(1);
	}
	setvbuf(f, NULL, _IOFBF, 32*1024);
	ctexscan(fn, lstfile, ctx_flag, f);
	fclose(f);

	if (ctx_flag) {
	    sprintf(buffer, "cct -q \"%s\" \"%s\"", temp_file, outfile);
	    do_cmd(buffer, True);
	}
	else {
	    if (local_cctspace) {
		sprintf(buffer, "cctspace %s \"%s\" \"%s\"",
				force_tilde ? "-t" : "", temp_file, outfile);
		do_cmd(buffer, True);
		rename_file(outfile, temp_file);
	    }
	    sprintf(buffer, "cctconv \"%s\" \"%s\"", temp_file, outfile);
	    do_cmd(buffer, True);
	}
	
	sprintf(buffer, "del \"%s\"", temp_file);
	message(buffer);
	unlink(temp_file);
#endif
    }

    /* write entries to .xref file */
    if (f_xref != NULL) {
	FILE *f_lst;
	char *p;
	int pass = 0;

	BASENAME((char *)fn, p);
	fprintf(f_xref, "%s:\n\t%s:", fn, p);
	for (pass = 0; pass < 2; pass++) {
	    if ((f_lst = fopen(lstfile, "rt")) != NULL) {
		setvbuf(f_lst, NULL, _IOFBF, 32*1024);
		while (True) {
		    if (fgets(buffer, sizeof(buffer), f_lst) == NULL) break;
		    if (fgets(outfile, sizeof(outfile), f_lst) == NULL) break;
		    if (buffer[0] == '\0' || outfile[0] == '\0') break;
		    if (buffer[strlen(buffer)-1] == '\n')
			buffer[strlen(buffer)-1] = '\0';
		    if (outfile[strlen(outfile)-1] == '\n')
			outfile[strlen(outfile)-1] = '\0';
		    if (buffer[0] == '\0' || outfile[0] == '\0') break;
		    BASENAME(buffer, p);
		    if (!pass)
			/* add filename to the referenced list */
			fprintf(f_xref, " %s", p);
		    else
			/* create entries for files not scanned */
			if (outfile[0] == '<')
			    fprintf(f_xref, "%s:\n\t%s:\n", buffer, p);
		}
	    }
	    fclose(f_lst);
	    if (!pass)
		fprintf(f_xref, "\n");
	}
    }

    /* recurse into list file */
    if (depth+1 < max_depth && !stat(lstfile, &buf) && buf.st_size > 3) {
	depth++;
#ifdef DEBUG
	fprintf(stderr, "Processing list file %s (depth=%d)\n", lstfile, depth);
#endif
	loop(lstfile);
	depth--;
    }
}

static void
loop(char *fn)
{
    char *fn_list=NULL;
    char *p;
    size_t fn_size=0;
    static char s[PATH_MAX], s1[PATH_MAX];

    if (!depth) {
	/* This is the main file, fn is the filename */
#ifdef DEBUG
	fprintf(stderr, "loop: main file \"%s\"\n", fn);
#endif
	BASENAME(fn, p)
#ifdef USE_TEMPDIR
	strcpy(s, p);
#else
	sprintf(s, CTEX_TMP "%s", p);
#endif
	p = get_file_ext(s);
	if (*p == '\0' || !fnstrcmp(p, ".ctx")) strcpy(p, ".tex");
	append_file(&fn_list, &fn_size, fn);
	append_file(&fn_list, &fn_size, s);
	append_file(&fn_list, &fn_size, "");	/* marks end of list */
    }
    else {
	/* fn is the list filename */
	FILE *f;
#ifdef DEBUG
	fprintf(stderr, "loop: list file \"%s\"\n", fn);
#endif
	if ((f=fopen(fn, "rt"))==NULL) goto end;
	setvbuf(f, NULL, _IOFBF, 32*1024);
	do {
	    /* read "inputfile" and "outputfile", each on a line */
	    if (fgets(s, sizeof(s), f) == NULL) break;
	    if (fgets(s1, sizeof(s1), f) == NULL) break;
	    if (s[0] == '\0' || s1[0] == '\0') break;
	    if (s[strlen(s)-1] == '\n') s[strlen(s)-1] = '\0';
	    if (s1[strlen(s1)-1] == '\n') s1[strlen(s1)-1] = '\0';
	    if (s[0] == '\0' || s1[0] == '\0') break;
#ifdef DEBUG
	    fprintf(stderr, "loop: src=\"%s\", dest=\"%s\"\n", s, s1);
#endif
	    if (!strcmp(s1, "<nop>"))
		continue;
	    if (!strcmp(s1, "<copy>")) {
		/* "/" as outputfile means verbatim copy (of graphics file) */
		BASENAME(s, p);
		process_file(s, p, True);
		continue;
	    }
	    append_file(&fn_list, &fn_size, s);
	    append_file(&fn_list, &fn_size, s1);
	} while (1);
	fclose(f);
	if (fn_list != NULL) {
	    /* marks end of list */
	    append_file(&fn_list, &fn_size, "");
	}
    }
    if (fn_list == NULL) goto end;

    fn=fn_list;
    while (*fn != '\0') {
	p = fn + strlen(fn) + 1;
	process_file(fn, p, False);
	fn = p + strlen(p) + 1;
    }

end:
    if (fn_list != NULL) free(fn_list);
    return;
}

static void
move_file_back(char *fn_base, char *ext)
{
    char s[PATH_MAX];
#ifdef USE_TEMPDIR

#  ifdef UNIX
    strcpy(s, "../");
#  else
    strcpy(s, "..\\");
#  endif
    strcpy(s + 3, fn_base);
    strcat(s + 3, ext);
    if (stat(s + 3, &buf)) return;
    rename_file(s + 3, s);
#else
    char *p;
    
    strcpy(s, fn_base);
    strcat(s, ext);
    p = s + sizeof(CTEX_TMP) - 1;
    if (stat(s, &buf)) return;
    rename_file(s, p);
#endif	/* USE_TEMPDIR */
}

static void patch_log_file(void)
{
    char fn_in[PATH_MAX], *fn_out;
    FILE *f_in, *f_out;

#ifdef DEBUG
    {
	int i;
	fprintf(stderr, "Filename mapping:\n");
	for (i = 0; i < fnmap_n; i++)
	    fprintf(stderr, "\t%s -> " CTEX_TMP "%s\n", fnmap[i].orig_fn,
				fnmap[i].temp_fn);
    }
#endif
    
    strcpy(fn_in, main_base);
    strcat(fn_in, ".log");
    fn_out = fn_in + sizeof(CTEX_TMP) - 1;
    
    if ((f_in = fopen(fn_in, "rb")) == NULL) return;
    setvbuf(f_in, NULL, _IOFBF, 32*1024);
    if ((f_out = fopen(fn_out, "w+b")) == NULL) {
	fprintf(stderr, "Cannot open or create %s, abort.\n", fn_out);
	exit(1);
    }
    setvbuf(f_out, NULL, _IOFBF, 32*1024);

    do {
	char *p;
#ifdef UNIX
	if (fgets(buffer, sizeof(buffer), f_in) == NULL) break;
#else
	for (p = buffer; p - buffer < sizeof(buffer) - 1;) {
#if 1
	    int c;
	    if ((c = fgetc(f_in)) == EOF || (*(p++) = c) == '\n') break;
#else
	    if (fread(p, 1, 1, f_in) != 1 || *(p++) == '\n') break;
#endif
	}
	if (p == buffer) break;
	*p = '\0';
#endif
	p = buffer;
	do {
	    char *q, ch;
	    size_t len;
	    fnmap_t *m, key = {NULL, NULL};
	    /* TODO: caseless search under MSDOS/Windows */
	    if ((p = strstr(p, CTEX_TMP)) == NULL)
		break;
	    /* look for the end of filename */
	    key.temp_fn = q = p + sizeof(CTEX_TMP) - 1;
	    while (*q != '\0' && !isspace(*q)
			    && strchr("()'\"$[]{};]", *q) == NULL)
		q++;
	    ch = *q;	/* save character */
	    *q = '\0';
	    m = bsearch(&key, fnmap, fnmap_n, sizeof(fnmap_t), fnmap_comp);
#ifdef DEBUG
	    fprintf(stderr, "patch_log_file: " CTEX_TMP "%s -> %s\n",
			key.temp_fn, m == NULL ? "(unchanged)" : m->orig_fn);
#endif
	    *q = ch;
	    if (m == NULL) {
		p = q;
		continue;
	    }
#ifdef UNIX
	    if (p - buffer >= 2 && !memcmp(p - 2, "./", 2))
		p -= 2;
#else
	    if (p - buffer >= 2 && (!memcmp(p - 2, "./", 2)
				    || !memcmp(p - 2, ".\\", 2)))
		p -= 2;
#endif
	    /* Warning: the new name might be longer */
	    len = strlen(m->orig_fn);
	    if (len == q - p) {
	        memcpy(p, m->orig_fn, len);
		p = q;
	    }
	    else if (len < q - p) {
	        memcpy(p, m->orig_fn, len);
	        strcpy(p + len, q);
		p = q;
	    }
	    else {
		size_t len1 = strlen(q);
		if (len1 + p - buffer + len > sizeof(buffer)) {
		    /* shouldn't happen */
		    fprintf(stderr, "Warning: line too long in .log file.\n");
		    p = q;
		    continue;
		}
		memmove(q + (len - (q - p)), q, len1 + 1);
		memcpy(p, m->orig_fn, len);
		p += len;
	    }
	} while (True);
	fprintf(f_out, "%s", buffer);
    } while (True);

    fclose(f_in);
    fclose(f_out);
#ifndef DEBUG
    unlink(fn_in);
#endif
}

static Boolean
parse_option(const char *str)
/* checks if the string pointed by p matches a CTeX option string.
 * returns True if the string is a CTeX option (and in this case the
 * option is processed), False otherwise. */
{
    OPTION *o;
    size_t l;
    int value, quote;
    const char *p;
    char *q;

    while (isspace(*str)) str++;
    if (*(str++) != '-') return False;
    for (o = options; o < options + NOPTIONS; o++) {
	l = strlen(o->name);
	if (fnstrncmp(o->name, str, l)) continue;
	p = str + l;
	if (*p != '\0' && *p != '=' && !isspace(*p)) continue;
       	if ((*p == '=') != (o->has_arg == True)) continue;
	if (o->var == NULL) continue;
	if (*(p+1) == '"' || *(p+1) == '\'') {
	    quote = *(++p);
	}
	else {
	    quote = -1;
	}
	switch (o->var_type) {
	    case VAR_INT:
		value = o->has_arg ? atoi(p+1) : o->value;
		*((int *)o->var) = value;
#ifdef DEBUG
		fprintf(stderr, "\tOption: %s = %d\n", o->help, value);
#endif
		break;
	    case VAR_BOOLEAN:
		value = o->has_arg ? atoi(p+1) : o->value;
		*((Boolean *)o->var) = value;
#ifdef DEBUG
		fprintf(stderr, "\tOption: %s = %s\n", o->help,
				value ? "true" : "false");
#endif
		break;
	    case VAR_STRING:
		if (*((char **)o->var) != NULL) free(*((char **)o->var));
		q = *((char **)o->var) = strdup(p+1);
		if (q == NULL) break;
		if (quote != -1) {
		    /* scan for closing quote */
		    while (*q != '\0' && *q != quote) {
			if (*q == '\\') q++;
			q++;
		    }
		    if (*q != quote) {
			fprintf(stderr, "Warning: missing closing quote in "
					"option \"\%s\".\n", str);
		    }
		    *q = '\0';
		}
		/* delete trailing newline */
		q = q + strlen(q) - 1;
		if (q >= *((char **)o->var) && *q == '\n') *(q--) = '\0';
		if (*(*((char **)o->var)) == '\0') {
		    free(*((char **)o->var));
		    *((char **)o->var) = NULL;
		}
#ifdef DEBUG
		fprintf(stderr, "\tOption: %s = %s\n", o->help,
				*((char**)o->var));
#endif
	    default:	/* do nothing */
		break;
	}
	return True;
    }
    return False;
}

static void
usage(char *argv0)
{
    OPTION *o;

    fprintf(stderr, "Usage: %s [options] [&fmt] input_file\n", argv0);
    fprintf(stderr, "Options:\n");
    for (o = options; o < options + NOPTIONS; o++) {
	sprintf(buffer, "  -%s", o->name);
	if (o->has_arg) {
	    switch (o->var_type) {
		case VAR_INT:
		    strcat(buffer, "=<number>");
		    break;
		case VAR_STRING:
		    strcat(buffer, "=\"<string>\"");
		    break;
		default:
		    break;
	    }
	}
	if (strlen(buffer) <= 28)
	    fprintf(stderr, "%-28s %s", buffer, o->help);
	else
	    fprintf(stderr, "%s\n%-28s %s", buffer, "", o->help);
	if (o->has_arg) {
	    switch (o->var_type) {
		case VAR_INT:
		    fprintf(stderr, " (default %d)", *((int *)o->var));
		    break;
		case VAR_STRING:
		    fprintf(stderr, " (default \"%s\")", 
			*((char **)o->var) == NULL ? "" : *((char **)o->var));
		    break;
		default:
		    break;
	    }
	}
	fprintf(stderr, ".\n");
    }
    fprintf(stderr, "All other options as well as the '&fmt' string "
		    "(if present) are passed to TeX.\n");
    fprintf(stderr, "Options may also be put on top of the "
		    "main document using the `%%&' convention.\n");
    exit(1);
}

static void
clean_temp(void)
{
    int ret = 0;
#if defined(GO32)
    struct ffblk sr;
    int i;

    i=findfirst(CTEX_TMP "*.*", &sr, /*FILE_ATTR*/0);
    while (!i && sr.ff_name[0] != '\0') {
#ifdef USE_TEMPDIR
	sprintf(buffer, CTEX_TMP "%s", sr.ff_name);
#else
	strcpy(buffer, sr.ff_name);
#endif
	if (verbose) fprintf(stderr, " Deleting \"%s\"\n", buffer);
	/* chmod(buffer, S_IREAD | S_IWRITE); */
	if (unlink(buffer)) {
	    ret = 1;
	    fprintf(stderr, " Warning: removing \"%s\" failed.\n", buffer);
	}
	i=findnext(&sr);
    }
#elif defined(WIN32)
    struct _finddata_t c_file;
    long hFile;

    if ((hFile = _findfirst(CTEX_TMP "*.*", &c_file)) != -1L) {
	while (True) {
	    if (!strcmp(c_file.name, ".") || !strcmp(c_file.name, ".."))
		goto next;
#ifdef USE_TEMPDIR
	    sprintf(buffer, CTEX_TMP "%s", c_file.name);
#else
	    strcpy(buffer, c_file.name);
#endif
	    /* skip subdirectories */
	    if (c_file.attrib & _A_SUBDIR) {
		fprintf(stderr, "Warning: skipping subdirectory %s\n", buffer);
		goto next;
	    }
	    if (verbose) fprintf(stderr, " Deleting \"%s\"\n", buffer);
	    if (_unlink(buffer))
		fprintf(stderr, "Warning: removing file %s failed!\n", buffer);
	next:
	    if(_findnext(hFile, &c_file) == -1) break;
	}
	_findclose(hFile);
    }
#elif USE_GLOB	/* UNIX or GO32, use glob() */
    glob_t g;
    int i;
    char *p;

    if (verbose) fprintf(stderr, "Cleaning temp files:\n");
#   ifdef UNIX
    if (glob(CTEX_TMP "*", GLOB_MARK | GLOB_PERIOD, NULL, &g)) exit(0);
#   else  /* GO32 */
    if (glob(CTEX_TMP "*", GLOB_MARK,               NULL, &g)) exit(0);
#   endif
    for (i = 0; i < g.gl_pathc; i++) {
	p = g.gl_pathv[i];
	if (!fnstrcmp(p, CTEX_TMP "./") ||
	    !fnstrcmp(p, CTEX_TMP "../")) continue;
	if (p[strlen(p) - 1] == '/' || p[strlen(p) - 1] == '\\') {
	    fprintf(stderr, " Warning: \"%s\" not removed.\n", p);
	    continue;
	}
	if (verbose) fprintf(stderr, " Deleting \"%s\"\n", p);
	if (unlink(p)) {
	    ret = 1;
	    fprintf(stderr, " Warning: removing \"%s\" failed.\n", p);
	}
    }
    globfree(&g);
#else
    system("/bin/rm -f "CTEX_TMP"*");
#endif	/* GO32, WIN32 */

#ifdef USE_TEMPDIR
    if (verbose) fprintf(stderr, " Deleting \""CTEX_TMP"\"\n");
    if (rmdir(CTEX_TMP)) {
	ret = 1;
	fprintf(stderr, " Warning: removing \""CTEX_TMP"\" failed.\n");
    }
#endif
    if (verbose) fprintf(stderr, "Done.\n");

    exit(ret);
}

static char cwd[PATH_MAX]=".";

static void
clean_up(void)
{
    if (temp_file != NULL) unlink(temp_file);
#ifndef UNIX
    /* restore CWD */
    chdir(cwd);
#endif
}

static Boolean comp_md5(char *ext)
/* Compute and MD5 sum of the file "main_base".ext,
 * the MD5 sum is set to 0 if any error (file inexist, read error, etc.).
 *
 * Then compare the MD5 sum with previously saved values, and replace the
 * saved values with the new values after comparison.
 *
 * Returns False if MD5 sums match, True if MD5 sums mismatch */
{
    extern int md5_stream(FILE *stream, void *resblock); /* in md5.c */
    static struct {
	char ext[4];
	unsigned char md5[16];
    } md5sums[16], *m;
    static int nmd5 = 0;

    FILE *f;
    char fname[PATH_MAX];
    unsigned char md5sum[16];
    int i;
    Boolean new_entry = False;

    sprintf(fname, "%s.%s", main_base, ext);
    if ((f = fopen(fname, "rb")) == NULL || md5_stream(f, md5sum))
	memset(md5sum, 0, sizeof(md5sum));
    if (f != NULL) {
	fclose(f);
	/* set MD5 sum to 0 if file size is 0 */
	if (stat(fname, &buf) || buf.st_size == 0)
	    memset(md5sum, 0, sizeof(md5sum));
    }

    /* lookup the saved MD5 sums */
    for (i=0; i < nmd5; i++)
	if (!fnstrcmp(md5sums[i].ext, ext)) break;

    if (i == nmd5) {
#ifdef DEBUG
	fprintf(stderr, "comp_md5: new entry %d for \"%s\".\n", i, ext);
#endif
	new_entry = True;
	if (i >= (sizeof(md5sums)/sizeof(md5sums[0]))) {
	    fprintf(stderr, "comp_md5: too many file extensions "
			    "(increase md5sums[]).\n");
	    exit(1);
	}
	nmd5++;
	strncpy(md5sums[i].ext, ext, sizeof(md5sums[i].ext));
	memset(md5sums[i].md5, 0, sizeof(md5sums[i].md5));
    }

    m = md5sums + i;

#ifdef DEBUG
    fprintf(stderr, "comp_md5: %s for %s.%s:\n\t",
	    new_entry ? "initial MD5 sum" : "old and new MD5 sums", 
	    main_base, ext);
    if (!new_entry) {
	for (i = 0; i < sizeof(md5sum); i++) fprintf(stderr, "%02x", m->md5[i]);
	fprintf(stderr, "\n\t");
    }
    for (i = 0; i < sizeof(md5sum); i++) fprintf(stderr, "%02x", md5sum[i]);
    fprintf(stderr, "\n");
#endif
    
    i = memcmp(m->md5, md5sum, sizeof(md5sum));
    memcpy(m->md5, md5sum, sizeof(md5sum));

    return i;
}

static Boolean
check_aux(char *ext)
/* compare timestamps of main_base.ext and main_file, returns True
 * if main_base.ext exists and is newer than main_file */
{
    char fn[PATH_MAX];

    sprintf(fn, "%s.%s", main_base, ext);
    return !stat(fn, &buf) &&
	   (stat(main_file, &buf1) || buf.st_mtime >= buf1.st_mtime);
}

int
main(int argc, char *argv[])
{
    int i, runs = 0;
    char src_name[PATH_MAX] = "";	/* src filename */
    char *name, *tex_opts = NULL, *p;
    FILE *f = NULL;			/* assigned a value just for making
					   gcc-2.8.1 happy */
    Boolean oldcct_flag = False, rerun = False, tex_errcode = 0;

    /* init some string options */
    gbk2uni_opts = strdup("-s");
#ifndef UNIX
    patchdvi_opts = strdup("-b");
#endif
    glossary_opts = strdup("-s gglo.ist");
    makeindex_prog = strdup("cctmkind");

    getcwd(cwd, sizeof(cwd));
    if (cwd[strlen(cwd)-1] == '\\') strcat(cwd, ".");
    atexit(clean_up);

    /* Determine action in Stage2 according to the name of the executable */
    BASENAME(argv[0], name);
    name = strdup(name);
    if (!fnstrcmp(p=get_file_ext(name), ".exe")) *p = '\0';
    /* strip the "cct" prefix */
    if (!fnstrncmp(name, "cct", 3))
	strcpy(name, name + 3);
    for (i=0; stage2_cmds[i] != NULL; i++)
	if (!fnstrcmp(name, stage2_cmds[i])) stage2 = i;
    free(name);

    /* Scan for input filename */
    name = NULL;
    for (i = 1; i < argc; i++) {
	if (argv[i][0] != '-' && argv[i][0] != '&') {
	    if (name != NULL) usage(argv[0]);
	    name = argv[i];
	}
    }

    /* Process options, unknown options (and "&fmt") are passed to TeX */
    tex_opts = strdup(TEX_OPTS);

    if (name != NULL) {
	/* Scan input file for '%&' commands */
	strcpy(src_name, name);
	/* Note ZLB: TeX first looks for `file.tex' then `file' (and 
	 * MikTeX will only look for `file.tex' if file does not have
	 * an extension). 
	 * We search the input file in a roughly compatible way */
#ifdef MIKTEX
	/* Only try to open the file when it has an extension */
	i = (*get_file_ext(src_name) != 0);
#else
	i = True;
#endif
	p = src_name + strlen(src_name);
	if (strcpy(p, ".ctx")  && (f = fopen(src_name, "rt")) == NULL &&
	    strcpy(p, ".tex")  && (f = fopen(src_name, "rt")) == NULL &&
	    strcpy(p, "") && i && (f = fopen(src_name, "rt")) == NULL);
	if (f == NULL) {
	    fprintf(stderr, "Cannot open input file.\n");
	    exit(1);
	}
	setvbuf(f, NULL, _IOFBF, 32*1024);
#ifdef DEBUG
	fprintf(stderr, "Process '%%&' lines in the file \"%s\".\n", src_name);
#endif
	while (True) {
	    int size = strlen(tex_opts);
	    if (fgets(buffer, sizeof(buffer), f) == NULL) break;
	    if (buffer[0] == '%') {
		if (buffer[1] == '&') {
		    int j = strlen(buffer+2);
		    while (j > 0 && isspace(buffer[2 + j - 1])) j--;
		    buffer[2 + j] = '\0';
		    if (!parse_option(buffer+2)) {
			/* 1 space for ' ' */
			tex_opts = realloc(tex_opts, size + j + 1 + 1);
			tex_opts[size++] = ' ';
			strcpy(tex_opts + size, buffer+2);
			size += j;
		    }
		}
	    }
	    else {
		break;	/* stop at first non-comment line */
	    }
	}
	fclose(f);
    }

    if (!fnstrcmp(get_file_ext(src_name), ".ctx")) ctx_flag = 1;

    /* Cmdline options */
#ifdef DEBUG
    fprintf(stderr, "Process options in the cmdline.\n");
#endif
    for (i = 1; i < argc; i++) {
	int size = strlen(tex_opts);
	if (argv[i][0] != '&' && argv[i][0] != '-') continue;
	if (!parse_option(argv[i])) {
	    int j = strlen(argv[i]);
	    tex_opts = realloc(tex_opts, size + j + 2);	/* 1 for ' ' */
	    tex_opts[size++] = ' ';
	    strcpy(tex_opts + size, argv[i]);
	    size += j;
	}
    }

    BASENAME(argv[0], p);
    if (verbose) fprintf(stderr, "%s, v%s, by ZLB.\n", p, CTEX_VERSION);

    if (help || (name == NULL && !clean)) usage(argv[0]);

    if (clean) clean_temp();

#ifndef UNIX
    if ((temp_file = mktemp(strdup("ctextemp_.XXXXXX"))) == NULL) {
#else
    if ((i = mkstemp(temp_file = strdup("ctextemp_.XXXXXX"))) == -1 || 
	close(i) == -1) {
#endif
	fprintf(stderr, "Cannot obtain temp filename, abort.\n");
	return 1;
    }

    /* No Stage3 for PDFTeX/PDFLaTeX */
    if (stage2 == PDFTeX || stage2 == PDFLaTeX) stage3 = NONE;

    if (stage3 != NONE) patchdvi = True;

Stage1:
    if (stage1 == NONE) goto Stage2;
    if (verbose) fprintf(stderr, "%s CTeX: Stage 1\n", SEP);
#ifdef USE_TEMPDIR
    strcpy(buffer, CTEX_TMP);
    buffer[strlen(buffer)-1] = '\0';	/* remove trailing '/' or '\\' */
    if (mkdir(CTEX_TMP, 0755) && errno != EEXIST) {
	fprintf(stderr, "Cannot make temp dir \""CTEX_TMP"\"\n");
	exit(2);
    }
#endif	/* USE_TEMPDIR */

    if (xref) {
	strcpy(buffer, src_name);
	strcpy(get_file_ext(buffer), ".xref");
	f_xref = fopen(buffer, "w+t");
	if (f_xref != NULL)
	    setvbuf(f_xref, NULL, _IOFBF, 32*1024);
    }

    loop(src_name);

Stage2:
    if (stage2 == NONE) goto Stage3;
    if (verbose) fprintf(stderr, "%s CTeX: Stage 2\n", SEP);
#ifdef USE_TEMPDIR
    message("cd " CTEX_TMP);
    if (chdir(CTEX_TMP)) {
	fprintf(stderr, "Cannot cd to \""CTEX_TMP"\", abort.\n");
	exit(1);
    }

#  ifndef MIKTEX
    /* set environment variable TEXINPUTS.
     * NOTE ZLB: no need to quote the pathnames even if the pathnames
     *		 contain spaces. */
    sprintf(buffer, "%s/" CTEX_TMP ":%s:%s", cwd, cwd,
		    (p=getenv("TEXINPUTS")) == NULL ? "" : p);
    if (verbose) fprintf(stderr, "+++ setenv TEXINPUTS %s\n", buffer);
    setenv("TEXINPUTS", buffer, 1);
#  else
    sprintf(buffer, " -include-directory=\"%s\"", cwd);
    tex_opts = realloc(tex_opts, strlen(tex_opts) + strlen(buffer) + 1);
    strcat(tex_opts, buffer);
#  endif
#endif	/* USE_TEMPDIR */

    if (strstr(stage2_cmds[stage2], "latex")) {
	/* LaTeX based formats: latex, pdflatex, amslatex, etc. */
	comp_md5("aux");
	comp_md5("idx");
	comp_md5("out");
	comp_md5("bbl");
#if defined(__GNUC__) && !defined(__MINGW32__)
	#warning FIXME: also check "toc", "lof", "lot", ... ?
#endif
	for (runs = 0; runs < latex_runs; runs++) {
	    sprintf(buffer, "%s %s \"%s\"", stage2_cmds[stage2], tex_opts,
			    main_file);
	    rerun = False;
	    tex_errcode = do_cmd(buffer, False);
#if 0
	    /* check the log file to determine whether to run latex again */
	    if (runs < latex_runs-1) {
		sprintf(buffer, "%s.log", main_base);
		f = fopen(buffer, "rt");
	    }
	    else
		f = NULL;
	    if (f != NULL)
		setvbuf(f, NULL, _IOFBF, 32*1024);
	    while (f != NULL) {
		int l;
		if (fgets(buffer, sizeof(buffer), f) == NULL) {
		    fclose(f);
		    break;
		}
		if (buffer[l = strlen(buffer)-1] == '\n') buffer[l] = '\0';
		if (strncmp(buffer, "LaTeX Warning: ", 15)) continue;
		if (!strcmp(buffer + 15, "Label(s) may have changed. "
			    "Rerun to get cross-references right.") /*||
		    !strcmp(buffer + 15, "There were undefined references.")*/)
		    rerun = True;
	    }
#endif
	    /* update .bbl file only if .aux file exists and is newer
	     * than the main file */
	    if (check_aux("aux")) {
		if (!runs && bibtex) {
		    sprintf(buffer, "bibtex %s \"%s\"",
			    bibtex_opts == NULL ? "" : bibtex_opts, main_base);
		    do_cmd(buffer, False);
		    if (comp_md5("bbl")) {
#ifdef DEBUG
			fprintf(stderr, "rerun = True (.bbl file changed).\n");
#endif
			rerun = True;
		    }
		}
		if (!rerun && comp_md5("aux")) {
#ifdef DEBUG
		    fprintf(stderr, "rerun=True (.aux file changed).\n");
#endif
		    rerun = True;
		}
	    }
	    /* update .out file only if it exists and is newer
	     * than the main file */
	    sprintf(buffer, "%s.out", main_base);
	    if (check_aux("out")) {
		sprintf(buffer, "gbk2uni %s \"%s.out\"",
				gbk2uni_opts == NULL ? "" : gbk2uni_opts,
				main_base);
		do_cmd(buffer, True);
		if (comp_md5("out")) {
#ifdef DEBUG
		    fprintf(stderr, "rerun=True (.out file changed).\n");
#endif
		    rerun = True;
		}
	    }
	/*--------------------------- makeindex ---------------------------*/
	for (i = 0; i < 2; i++) {
	    static char *src_ext[] = {"idx", "glo"};
	    static char *dst_ext[] = {"ind", "gls"};
	    char *opts[] = {makeindex_opts, glossary_opts};

	    /* update the index file only if it exists and is newer
	     * than the main file */
	    if (check_aux(src_ext[i])) {
		FILE *f;
		if (ctx_flag) {
		    /* uncct the index file */
		    sprintf(buffer, "uncct \"%s.%s\" \"%s\"",
				    main_base, src_ext[i], temp_file);
		    do_cmd(buffer, True);
		    sprintf(buffer, "%s.%s", main_base, src_ext[i]);
		    rename_file(temp_file, buffer);
		}
		/* append the cmdline to index file */
		sprintf(buffer, "%s.%s", main_base, src_ext[i]);
		if ((f = fopen(buffer, "at")) != NULL) {
		    sprintf(buffer, "%s -o %s.%s %s \"%s.%s\"",
				makeindex_prog,
				main_base, dst_ext[i], 
				(opts[i] != NULL) ? opts[i] : "", 
				main_base, src_ext[i]);
		    fprintf(f, "\n%% %s\n", buffer);
		    fclose(f);
		}
	 	if ((!check_aux(dst_ext[i]) || comp_md5(src_ext[i]))) {
#ifdef DEBUG
		    fprintf(stderr, "rerun=True (.%s file changed)\n",
				    src_ext[i]);
#endif
		    rerun = True;
		    do_cmd(buffer, True);
#if 0
		    if (ctx_flag) {
			/* cct the .ind file */
			sprintf(buffer, "cct \"%s.%s\" \"%s\"",
				    main_base, dst_ext[i], temp_file);
			do_cmd(buffer, True);
			sprintf(buffer, "%s.%s", main_base, dst_ext[i]);
			rename_file(temp_file, buffer);
		    }
#endif
		}
	    }
	}
	/*------------------------- end of makeindex ----------------------*/
	    if (!rerun || tex_errcode) {
		runs++;
		break;
	    }
	}
    }
    else {
	sprintf(buffer, "%s %s \"%s\"", stage2_cmds[stage2], tex_opts,
			main_file);
	rerun = False;
	runs = 1;
	tex_errcode = do_cmd(buffer, False);
    }

    oldcct_flag = (strncmp(stage2_cmds[stage2], "pdf", 3) &&
			patchdvi_check(main_base));
    if (strncmp(stage2_cmds[stage2], "pdf", 3) && patchdvi && oldcct_flag) {
#ifndef UNIX
	char *p;
	/* set environment variables (don't override existing ones) */
	setenv("CCHZPATH", CCHZPATH, 0);
	setenv("CCPKPATH", CCPKPATH, 0);

	/* For dvips */
	sprintf(buffer, "%s:%s", getenv("CCPKPATH"),
			(p=getenv("TEXPKS")) == NULL ? "" : p);
	setenv("TEXPKS", buffer, 1);

	sprintf(buffer, "patchdvi %s -q -y -r%d \"%s.dvi\" "CTEX_TMP".dvi",
			patchdvi_opts, dpi, main_base);
	do_cmd(buffer, True);
	sprintf(buffer, "%s.dvi", main_base);
	rename_file(CTEX_TMP".dvi", buffer);
#endif	/* !defined(UNIX) */
    }

    move_file_back(main_base, ".pdf");

    if (!fnmap_n) {
	move_file_back(main_base, ".dvi");
	move_file_back(main_base, ".log");
    }
    else {
	qsort(fnmap,  fnmap_n,  sizeof(fnmap_t), fnmap_comp);

	if (strncmp(stage2_cmds[stage2], "pdf", 3)) {
	    sprintf(buffer, "%s.dvi", main_base);
	    p = buffer + sizeof(CTEX_TMP) - 1;
	    if (verbose) {
    		fprintf(stderr, "+++ Converting \"%s\" --> \"%s\"\n", buffer,p);
	    }
	    if (patch_dvi_file(buffer, p, fnmap, fnmap_n) == 0)
		unlink(buffer);
	    else
		move_file_back(main_base, ".dvi");
	}

	if (verbose) {
    	    fprintf(stderr, "+++ Converting \"%s.log\" --> \"%s.log\"\n", 
			    main_base, main_base + sizeof(CTEX_TMP) - 1);
	}
	patch_log_file();
    }

#ifdef USE_TEMPDIR
    message("cd ..");
    chdir("..");
#else
    strcpy(main_base, main_base + sizeof(CTEX_TMP) - 1);
    strcpy(main_file, main_file + sizeof(CTEX_TMP) - 1);
#endif	/* USE_TEMPDIR */
    
Stage3:
    if (stage3 == NONE) goto End;
    if (verbose) fprintf(stderr, "%s CTeX: Stage 3\n", SEP);

    switch (stage3) {
	case DVIPS:
	    /* Note ZLB: PATH of PK fonts is appended to TEKPKS in Stage 2 */
#ifdef UNIX
	    sprintf(buffer, "%s -D %d %s \"%s\"", 
			    oldcct_flag ? "dvipsc" : "dvips", dpi,
			    dvips_opts == NULL ? "" : dvips_opts, main_base);
#else
	    sprintf(buffer, "dvips -D %d \"%s\"", dpi, main_base);
#endif
	    break;
	case DVIPDFMX:
	    if (oldcct_flag) {
#if defined(__GNUC__) && !defined(__MINGW32__)
		#warning TODO: use dvipsc + ps2pdf here
#endif
		fprintf(stderr, "Warning: CCT's dvi file format detected, "
				"dvipdfmx not executed\n");
		buffer[0] = '\0';
		break;
	    }
	    sprintf(buffer, "dvipdfmx -r %d %s \"%s\"", dpi, 
			    dvipdfmx_opts == NULL ? "" : dvipdfmx_opts,
			    main_base);
	    break;
    }
    if (buffer[0] != '\0') do_cmd(buffer, True);

End:

    if (verbose) {
	fprintf(stderr, "\nStatistics:\n");
	fprintf(stderr, "  Input file(s) preprocessed by `%s'.\n",
#ifdef UNIX
			ctx_flag ? "cct" : 
#else
			ctx_flag ? "cct.exe" : 
#endif
			    cctspace ? "cctspace | cctconv" : "cctconv");
	if (runs > 1) {
	    fprintf(stderr, "  '%s' has been run %d times.\n", 
			    stage2_cmds[stage2], runs);
	}

	if (gbk2uni_opts != NULL)
	    fprintf(stderr, "  gbk2uni options:   `%s'\n", gbk2uni_opts);
	if (bibtex_opts != NULL)
	    fprintf(stderr, "  bibtex options:    `%s'\n", bibtex_opts);
	if (makeindex_opts != NULL)
	    fprintf(stderr, "  makeindex options: `%s'\n", makeindex_opts);
	if (glossary_opts != NULL)
	    fprintf(stderr, "  glossary options: `%s'\n", glossary_opts);
	if (stage3 == DVIPS)
	    fprintf(stderr, "  dvips options:     `-D %d %s'\n",
			dpi, dvips_opts == NULL ? "" : dvips_opts);
	if (stage3 == DVIPDFMX)
	    fprintf(stderr, "  dvipdfmx options:  `-r %d %s'\n",
			dpi, dvipdfmx_opts == NULL ? "" : dvipdfmx_opts);
    }

    if (f_xref != NULL) {
	fprintf(stderr, "  cross references written in \"%s.xref\".\n",
			main_base);
	fclose(f_xref);
    }
    
    
    if (tex_errcode)
	fprintf(stderr, "\nWarning: exit code (%d) of \"%s\" is not 0.\n",
			tex_errcode, stage2_cmds[stage2]);

    if (rerun) {
	BASENAME(argv[0], name);
	fprintf(stderr, "\nWarning: you may need to run \"%s\" again "
			"to get cross references right.\n", name);
    }

    if (verbose) fprintf(stderr, "\n");

    return 0;

    goto Stage1;	/* to make GCC happy */
}
