/* $Id: text2dvi.c,v 1.5 2004/11/26 14:11:25 zlb Exp $ */

/* This program transforms a plain text file into a DVI file.

   The font used for ASCII characters is CMTT10 at 10pt whose default
   width is equal to 5.249958pt.

  Author: Zhang Linbo.
*/

typedef unsigned char byte;

static int CJK_flag = 0;

#define ASCII_FONT_NAME "cmtt10"	/* name of the ASCII font used */
#define ASCII_FONT_DS	10.0		/* design size */
#define HALF_CHAR_WIDTH	5.249958	/* width of the characters */
#define CC_SIZE		9.636		/* width of Chinese characters in pt */
#define LINE_SPACING	12		/* line spacing */

#define FOOTSKIP        ((long)(2.0*LINE_SPACING*65536.0+0.5))
#define FOOTHEIGHT	((long)(1.0*LINE_SPACING*65536.0+0.5))

#define DVI_ID		2		/* DVI id byte */
#define	NUM		25400000l	/* DVI numerator */
#define DEN		473628672l	/* DVI denominator */
#define MAG		1000l		/* magnification */

int chars_per_line;			/* # of chars/line */
int lines_per_page;			/* # of lines/page */
int ziti;				/* style */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

byte s_in[256], s_out[256+4];
FILE *f_in,*f_out;

int posx,posy;

long hsize,vsize,vsize1;
long h,v;			/* current position */
long w,x,y;			/* w=half char width, x=2*w-CC_SIZE, y=line spacing */
int  pageno0,np,pageno_flag,fontno,bop_flag;
long ll,loc;

void put_ascii_char(byte ch);

void write_dvi_number(unsigned long value, int len, FILE *f_dvi)
/* write a number of 'len' bytes (len<=4) to dvi file */
{
  byte buffer[4];
  int i;

  for (i=0; i<len; i++) {
    buffer[len-1-i]=(byte)(value%256);
    value/=256;
  }
  fwrite(buffer, len, 1, f_dvi);
}

void set_font(int no)
/* set current font */
{
  if (fontno == no) return;
  if (no < 64)
    fputc(171+(fontno=no),f_out);
  else {
    fputc(235,f_out);
    fputc(fontno=no,f_out);
  }
}

void ascii_font_def(int postamble)
/* output ascii font definition */
{
  register byte *p;
  register int i;
  static int ascii_flag = 0;

  if ((!postamble && ascii_flag) || (postamble && !ascii_flag)) return;

  fputc(243,f_out);			/* fnt_def1 */
  fputc(0,f_out);			/* no. 0 for ASCII font */
  write_dvi_number(0l, 4, f_out);	/* check-sum */
  ll=(long)(ASCII_FONT_DS*65536.0+0.5);
  write_dvi_number(ll, 4, f_out); 	/* scale size */
  write_dvi_number(ll, 4, f_out);	/* design size */
  p=ASCII_FONT_NAME; i=strlen(p);
  fputc(0,f_out);			/* a=0 */
  fputc(i,f_out);			/* l=strlen(p) */
  fwrite(p,1,i,f_out);			/* font name */
  ascii_flag=1;
}

static void cc_font_def0(int fontno, char *fontname, int suffix_no)
{
  char buffer[64], *p;

  if (suffix_no < 0) {
    p = fontname;
  } else {
    sprintf(buffer, "%s%02d", fontname, suffix_no);
    p = buffer;
  }

  fputc(243,f_out);			/* fnt_def1 */
  fputc(fontno,f_out);			/* fontno */
  if (!CJK_flag) 
    ll=(long)(CC_SIZE*1048576.0+0.5);	/* check-sum=char height */
  else
    ll=0;
  write_dvi_number(ll, 4, f_out);
  ll=(long)(CC_SIZE*65536.0+0.5);	/* design-size=scale_size=char width */
  write_dvi_number(ll, 4, f_out);	/* scale-size */
  if (CJK_flag)
    ll=655360;
  write_dvi_number(ll, 4, f_out);
  fputc(0,f_out);			/* a=0 */
  fputc(strlen(p),f_out);		/* l=strlen(p) */
  fwrite(p,1,strlen(p),f_out);		/* font name */
}

int cc_font_def(int fno)
/* output Chinese font definition, returns the fontno.
 * fno is the font suffix of the GBK font (1-94), 0 means postamble */
{
  register int fontno = 1;
  static char *CJK_ziti[] = {"gbksong", "gbkhei", "gbkkai", "gbkfs"};
  static int fontcount = -1;
  static int CJK_fonts[94];

  if (!CJK_flag) {
    if (fno > 0) {
      /* not postamble */
      if (fontcount <= 0) {
        char p[]="ZiTi=A";
	p[5]+=ziti;
        cc_font_def0(fontcount = 1, "ccfnta", -1);
        /* special command selecting Chinese character's style */
        fputc(239,f_out);	/* xxx1 */
        fputc(strlen(p),f_out);
        fwrite(p,1,strlen(p),f_out);
      }
    } else {
      /* postamble */
      if (fontcount > 0) {
        cc_font_def0(fontcount = 1, "ccfnta", -1);
      }
    }
  } else {
    if (fno > 0) {
      /* not postamble */
      if (fontcount == -1) {
	fontcount = 0;
	memset(CJK_fonts, 0, sizeof(CJK_fonts));
      } else {
	if (CJK_fonts[fno - 1] > 0) return CJK_fonts[fno - 1];
      }
      fontno = CJK_fonts[fno - 1] = ++fontcount;
      cc_font_def0(fontno, CJK_ziti[ziti], fno);
    } else {
      /* postamble */
      int i;
      for (i = 0; i < sizeof(CJK_fonts)/sizeof(CJK_fonts[0]); i++) {
	if (CJK_fonts[i] > 0)
	  cc_font_def0(CJK_fonts[i], CJK_ziti[ziti], i + 1);
      }
    }
  }

  return fontno;
}

void bop(void)
/* begin a new page */
{
  register int i;
  long loc0;

  np++;
  loc0=loc; loc=ftell(f_out);
  fputc(139,f_out);				/* bop command */
  ll=np+pageno0;
  for (i=0;i<10;i++) {				/* page numbers */
    write_dvi_number(ll, 4, f_out);
    ll=0l;
  }
  write_dvi_number(loc0, 4, f_out);		/* previous page loc. */
  /* define w */
  fputc(151,f_out);				/* w4 command */
  write_dvi_number(w, 4, f_out);
  /* define x */
  fputc(156,f_out);				/* x4 command */
  write_dvi_number(x, 4, f_out);
  /* reset horizontal position */
  fputc(146,f_out);				/* right4 command */
  write_dvi_number(-(w+x), 4, f_out);
  /* define y */
  fputc(165,f_out);				/* y4 command */
  write_dvi_number(y, 4, f_out);
  /* save current position */
  fputc(141,f_out);				/* push command */
  h=0l;
  v=y;
  bop_flag=1;
  fontno=-1;
}

void eop(void)
/* mark end of a page */
{
  if (!bop_flag) return;
  printf("[%d]",np+pageno0);
  if (pageno_flag) {
    byte s[128];
    register byte *p=s;
    register int i;
    long ll;
    sprintf(s,"%d",np+pageno0); i=strlen(s);
    /* move current position */
    fputc(146,f_out);				/* right4 command */
    ll=(hsize-i*w)/2-h; h+=ll;
    write_dvi_number(ll, 4, f_out);
    fputc(165,f_out);				/* y4 command */
    ll=vsize+FOOTSKIP-v; v+=ll;
    write_dvi_number(ll, 4, f_out);
    while (i--) put_ascii_char(*(p++));
  }
  bop_flag=0;
  fputc(142,f_out);				/* pop command */
  fputc(140,f_out);				/* eop command */
}

void advance_line(void)
/* advance to next line */
{
  if (!bop_flag) bop();
  fputc(142,f_out);				/* pop command */
  fputc(161,f_out);				/* y0 command */
  fputc(141,f_out);				/* push command */
  h=0l;
  if ((v+=y)>vsize) eop();			/* start a new page */
}

void put_ascii_char(byte ch)
/* output an ASCII char */
{
  if (!bop_flag) bop();
  ascii_font_def(0);
  if ((h+w)>hsize) advance_line();
  if (!bop_flag) bop();
  if (ch!=' ') {
    set_font(0);
    fputc(ch,f_out);	 			/* set_char command */
  } else {
    fputc(147,f_out);				/* w0 command */
  }
  h+=w;
}

void put_cc_char(byte ch1,byte ch2)
/* output an ASCII char */
{
  int fno, cno;

  if (!CJK_flag) {
    fno = 1;
    cno = 0;	/* this makes gcc happy */
  } else {
    int no = (ch1 - 0x81) * 190 + ((ch2 >= 0x7F) ? (ch2 - 1) : ch2) - 64;
    fno = no / 256 + 1;
    cno = no & 255;
  }
  
  if (!bop_flag) bop();
  fno = cc_font_def(fno);
  if ((h+w+w)>hsize) advance_line();
  if (!bop_flag) bop();
  set_font(fno);
  if (!CJK_flag) {
    fputc(ch1-128,f_out); 			/* set_char command */
    fputc(ch2-128,f_out); 			/* set_char command */
  } else {
    if (cno < 128)
      fputc(cno,f_out); 			/* set_char command */
    else {
      fputc(128,f_out); 			/* set1 command */
      fputc(cno,f_out);
    }
  }
  fputc(152,f_out);				/* x0 command */
  h+=w+w;
}

void *sp2a(long dim,byte s[])
/* transform a dimension in sp to cm in string form */
{
  long d=32768l*7227l;
  long mod,l;
  register int i,j,k,n;

  /* compute dim*127/d */
  n=(int)(127l*(dim/d)); mod=dim%d;
  l=mod;
  for (i=1;i<127;i++) {
    mod+=l;
    if (mod>=d) {n++; mod-=d;}
  }
  sprintf(s,"%d.",n);
  k=strlen(s); strcat(s,"0000");
  for (j=0;j<4;j++) {
    n=0;
    mod=mod+mod;
    if (mod>=d) {n+=5; mod-=d;}
    l=mod+mod;
    if (l>=d) {n+=2; l-=d;}
    l+=l;
    if (l>=d) {n++; l-=d;}
    mod+=l;
    if (mod>=d) {n++; mod-=d;}
    s[k++]+=n;
  }
  return(s);
}

void usage(void)
{
  fprintf(stderr, "Usage: text2dvi [options] inputfile [outputfile.dvi]\n");
  fprintf(stderr, "Options:\n");
  fprintf(stderr, "  -CJK\tuse CJK fonts (default: use CCT fonts).\n");
  fprintf(stderr, "  -c#\tnumber of chars/line (default 80).\n");
  fprintf(stderr, "  -l#\tnumber of lines/page (default 58).\n");
  fprintf(stderr, "  -z#\ttypeface (0=song, 1=hei, 2=kai, 3=fangsong).\n");
  fprintf(stderr, "  -p \tdon't add page numbers.\n");
  fprintf(stderr, "  -n#\tfirst page number (default 1).\n");
  exit(0);
}

static byte *buffer = NULL;
static size_t buffer_size = 0, buffer_count = 0;

static void save_ascii_char(byte ch)
{
    if (buffer_count + 1 >= buffer_size)
	buffer = realloc(buffer, buffer_size += 1024);
    buffer[buffer_count++] = ch;
    buffer[buffer_count] = '\0';
}

static void save_cc_char(byte ch1, byte ch2)
{
    if (buffer_count + 2 >= buffer_size)
	buffer = realloc(buffer, buffer_size += 1024);
    buffer[buffer_count++] = ch1;
    buffer[buffer_count++] = ch2;
    buffer[buffer_count] = '\0';
}

static void flush_buffer(void)
{
    byte *p = buffer, *q, *q0, ch;

    while (p - buffer < buffer_count) {
	if (buffer_count - (p - buffer) <= chars_per_line) {
	    while ((ch = *(p++)) != '\0') {
		(ch < 129) ? put_ascii_char(ch) : put_cc_char(ch, *(p++));
	    }
	    buffer_count = 0;
	    return;
	}
	q = q0 = p;
	/* find line break position */
	while (q - p < chars_per_line) {
	    if ((ch = *q) < 129) {
		if (isspace(ch = *q)) q0 = q;
		q++;
	    } else {
		q0 = q++;
		q++;
	    }
	}
	if (q0 == p) q0 = buffer + chars_per_line;
	while (p < q0) {
	    ch = *(p++);
	    (ch < 129) ? put_ascii_char(ch) : put_cc_char(ch, *(p++));
	}
	if (p - buffer >= buffer_count) {
	    buffer_count = 0;
	    return;
	}
	advance_line();
	while (isspace(*p)) p++;	/* skip leading spaces */
    }
}

int main(int argc, char *argv[])
{
  register int i,n;
  int k;
  byte *p,s[256];

  printf("TEXT2DVI V1.2, Nov. 2004. Author: Zhang Linbo\n");
  printf("This program transforms a plain text file to DVI format.\n");

  f_in=f_out=NULL;
  s_in[0]=s_out[0]='\0';
  chars_per_line=80;
  lines_per_page=58;
  ziti=0;
  pageno_flag=1;
  pageno0=0;

  for (i=1; i<argc; i++) {
    if (argv[i][0]=='-') switch (toupper(argv[i][1])) {
      case 'C':
	if (toupper(argv[i][2]) == 'J' && toupper(argv[i][3]) == 'K' &&
		argv[i][4] == '\0') {
	  CJK_flag = 1;
	} else {
          chars_per_line=atoi(argv[i]+2);
          if (chars_per_line<=0) {
      opt_err:
            fprintf(stderr, "Invalid value in \"%s\"!\n", argv[i]);
            usage();
          }
	}
        break;
      case 'L':
        lines_per_page=atoi(argv[i]+2);
        if (lines_per_page<=0) goto opt_err;
        break;
      case 'Z':
        ziti=atoi(argv[i]+2);
        if (ziti<0 || ziti>=4) goto opt_err;
	break;
      case 'P':
        pageno_flag=0;
        break;
      case 'N':
        pageno0=atoi(argv[i]+2)-1;
        break;
      default:
        fprintf(stderr, "Invalid option \"-%c\".\n", argv[i][1]);
        usage();
    } else {
      if (s_in[0]=='\0') {
	strncpy(s_in, argv[i], sizeof(s_in)-1);
        s_in[sizeof(s_in)-1]='\0';
      } else if (s_out[0]=='\0') {
	strncpy(s_out, argv[i], sizeof(s_out)-1);
        s_out[sizeof(s_out)-1]='\0';
      } else {
	fprintf(stderr, "Error: third filename specified (%s).\n", argv[i]);
	usage();
      }
    }
  }

  if (s_in[0]=='\0') usage();

  if ((f_in=fopen(s_in,"rt"))==NULL) {
    printf("Cannot open input file \"%s\"!\n",s_in); goto fin;
  }

  if (s_out[0]=='\0') {
    strcpy(s_out,s_in);
    i=strlen(s_out)-1; if (i<0) goto fin;
    for (;i>=0 && s_out[i]!='.';i--);
    if (i>=0 && s_out[i+1]!='\\') s_out[i]='\0';
    strcat(s_out,".dvi");
  }

  if (!strcmp(s_in,s_out)) {
    printf("Input & output file names are the same!\n");
    goto fin;
  }

  if ((f_out=fopen(s_out,"w+b"))==NULL) {
    printf("Cannot open output file \"%s\"!\n",s_out); goto fin;
  }
  if (setvbuf(f_out,NULL,_IOFBF,30*1024)) {
    printf("Cannot allocate output buffer!\n"); goto fin;
  }

  w=(long)(HALF_CHAR_WIDTH*65536.0+0.5);
  x=w+w-(long)(CC_SIZE*65536.0+0.5);
  y=(long)(LINE_SPACING*65536.0+0.5);
  hsize=w*chars_per_line;
  vsize=y*lines_per_page+y; vsize1=vsize+FOOTHEIGHT;

  printf("Transforming %s ==> %s...\n",s_in,s_out);
  printf("Page width=%scm, Page height=%scm\n",
         (char *)sp2a(hsize,s),
	 (char *)sp2a(pageno_flag ? vsize1:vsize,s+128));

  /* create DVI preamble */
  fputc(247,f_out);				/* pre command */
  fputc(DVI_ID,f_out);				/* DVI id. */
  write_dvi_number(NUM, 4, f_out);		/* num */
  write_dvi_number(DEN, 4, f_out);		/* den */
  write_dvi_number(MAG, 4, f_out);		/* mag */
  p=" TEXT2DVI V1.1"; i=strlen(p);
  fputc(i,f_out); fwrite(p,1,i,f_out);		/* comments */

  loc=0xffffffffl;
  bop_flag=0;
  np=0; n=0; k=0;
  /* processing pages */
  do {
    i=fgetc(f_in); if (i==EOF /*|| i==0x1a*/) break;
    if (i==13) continue;
cont:
    if (i>=32 && i<=127) {
      save_ascii_char((byte)i);			/* ASCII char */
      n++;
    } else if (i>=129 && i<=254) {
      byte ch;
      ch=i; i=fgetc(f_in);
      if (i==EOF || i==0x1a) break;
      if (i<64 || i>254 || i == 127) {
        printf("\nIncomplete Chinese character code ignored, line %d\n",k+1);
        goto cont;
      }
      save_cc_char(ch,(byte)i);			/* Chinese char */
      n+=2;
    } else if (i=='\t') {
      do save_ascii_char(' '); while ((++n)&7);
    } else if (i==12) {
      flush_buffer();
      eop();					/* new page */
      n=0;
    } else if (i=='\n') {
      flush_buffer();
      advance_line();				/* new line */
      n=0; k++;
    } else {
      printf("\nInvalid character (%d) ignored at line %d\n",i,k+1);
    }
  } while (1);

  if (bop_flag) {flush_buffer(); eop();}

  /* create the postamble */
  ll=loc; loc=ftell(f_out);
  fputc(248,f_out);				/* post command */
  write_dvi_number(ll, 4, f_out);		/* last page loc. */
  write_dvi_number(NUM, 4, f_out);		/* num */
  write_dvi_number(DEN, 4, f_out);		/* den */
  write_dvi_number(MAG, 4, f_out);		/* mag */
  ll=pageno_flag ? vsize1:vsize;
  write_dvi_number(ll, 4, f_out);		/* maxi. page height */
  ll=hsize; write_dvi_number(ll, 4, f_out);	/* maxi. page width */
  write_dvi_number(1, 2, f_out);		/* stack depth */
  write_dvi_number(np, 2, f_out);		/* # of pages */
  ascii_font_def(1);				/* ASCII font definition */
  cc_font_def(-1);				/* Chinese font definition */
  /* the post-post command */
  fputc(249,f_out);				/* post-post command */
  write_dvi_number(loc, 4, f_out);		/* loc. of the postamble */
  fputc(DVI_ID,f_out);				/* DVI id */
  for (i=0;i<4;i++) fputc(223,f_out);

  /* adjust file length to multiple of 4 */
  loc=ftell(f_out);
  n=(int)(loc&(4l-1));
  if (n) {
    n=4-n;
    for (i=0;i<n;i++) fputc(223,f_out);
  }

  printf("\nDone!\n");

fin:
  if (f_in!=NULL) fclose(f_in);
  if (f_out!=NULL) fclose(f_out);
  return 0;
}
