/* $Id: patchdvi.c,v 1.2 2004/11/26 11:41:33 zlb Exp $ */

/* Maps Chinese characters in a dvi file to temperary PK fonts.
 *
 * Change log:
 *
 * 2003-02-19: (v1.15)
 *
 * - stricmp/strnicmp: segmentation fault if str len >= 1024 fixed.
 *
 * 2003-05-01: (v1.16)
 *
 * - Mingw32 compiler support
 * - '-p' option for Unix
 *
 * 2003-05-09: (v1.17)
 *
 * - remove 'ctextemp_' prefix in src specials
 * - write usage message to stdout instead of stderr
 * - change 'src:###file' or 'src:###file.tex' to 'src:###file.ctx' if
 *   'file.ctx' exists.
 *
 * 2003-06-23: (v1.18)
 *
 * - comment out code on filename mapping (ctextemp_* ==> *) because
 *   it is handled by ctex.exe
 *
 * 2004-02-28: (v1.19)
 *
 * - add '-b2' command-line option
 * 
 * */

#ifndef GENPK
#  define GENPK	1
#endif

typedef unsigned char byte;

static byte *version="PATCHDVI v1.18 by ZHANG Linbo, Feb 2004.";

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <sys/types.h>
#if defined(WIN32) && !defined(__MINGW32__)
#  define PATH_MAX      _MAX_PATH+1
#else
#  include <limits.h>   /* PATH_MAX */
#endif

#if !defined(UNIX) && !defined(GO32)
#  include <process.h>
#  include <dos.h>
# ifdef WIN32
#  include <malloc.h>
#  include <direct.h>
# else
#  include <alloc.h>
#  include <dir.h>
# endif
#else
# ifndef WIN32
#  include <unistd.h>
#  include <pwd.h>
# endif
#endif

#if defined(UNIX) || defined(GO32) || defined(WIN32)
#  define far
#  define farmalloc	malloc
#  define farfree	free
#  define near
#  define huge
#  define long 		int
#endif

#include "proto.h"
#include "patchdvi.h"
#include "string-utils.h"

static byte *buffer, *pkfunc_buffer=NULL;

#define LSB	0
#define MSB	1
static int byte_order;

static long maxccheight=0l;

static int yes_flag=0, quiet=0;

static double dpix=0.0, dpiy=0.0;
static double ss0,ds0=10.0;	/* design size of tmp pk fonts (pt) */
static int pk_flag=1;
static int bmf_flag=0;		/* whether perform BMF==>PCX/PNG
				   (1=xdvi+PCX, 2=dviwin+PCX, 3=xdvi+PNG) */
static int dvistack_adjust=0;
#ifndef UNIX
  static byte pk_name[PATH_MAX]="tmp";
#else
  static byte pk_name[PATH_MAX]="";
#endif
static int code0=0, code1=127, ncodes=128;

static unsigned long num, den, mag=0;
static FILE *f_in=NULL, *f_out=NULL;
static byte s_in[PATH_MAX], s_out[PATH_MAX];
static long ploc;		/* location of previous page */
static int pass;
static int postamble;   	/* true if processing postamble */

static int ccdummy=-1;		/* ccdummy font number */
static fontdescr *froot=NULL;	/* list of "ccfnt" defs */
static ccfontdescr *croot=NULL;  /* list of Chinese chars */
static ccdescr huge *cclist=NULL;
static int ziti;
static int hzmode, infontno, outfontno;
static int ccsave=-1;
static int nbrofccfonts;
static int nbroftmpfonts;
static long totalcc;
static unsigned *ccfontnumbers;

static byte hzpath[PATH_MAX];
static byte pkpath[PATH_MAX];

static int is_ky=0, hanzi0=HanZi0, hanzi1=HanZi1, hanzi2=HanZi2,
                    maxccno=MAXCCNO;

#define ZITI_STACK_DEPTH	16
static struct {
  int ziti;
} ziti_stack0[ZITI_STACK_DEPTH], *ziti_stack=ziti_stack0;

#define ErrExit(s)					\
{							\
	fprintf(stderr,"%s (file \"%s\", line %d)\n",	\
			s, __FILE__, __LINE__);		\
	if (f_in!=NULL) fclose(f_in);			\
	if (f_out!=NULL) fclose(f_out);			\
	exit(1);					\
}

#define Warning(s)	warning(s,__LINE__)

void warning _P2(byte *,s, int,line)
{
  fprintf(stderr,"Warning: %s (patchdvi.c: line %d)\n",s,line);
}

void usage _VOID
{
  printf("\n%s\n", version);
  printf("\nUsage:\n\tC>patchdvi [options] infile[.dvi] [options] outfile[.dvi] [options]\n\n");
  printf("Valid options:\n");
  printf("    -y                - Yes to all questions\n");
  printf("    -r<dpix>[x<dpiy>] - Specify desired resolution (default: -r300x300)\n");
  printf("    -m<mag>           - Change overall magnification of the dvi file.\n");
  printf("    -f                - Don't create PK fonts for Chinese characters.\n");
  printf("    -b,-b0            - Replace \"\\special{BMF=file.bmf}\" with\n");
  printf("                        \"\\special{em:graph file.pcx ###in ###in}\" (xdvi,emTeX).\n");
  printf("    -b1               - Replace \"\\special{BMF=file.bmf}\" with\n");
  printf("                        \"\\special{isoscale file.pcx, ####in ####in}\" (dviwin).\n");
  printf("    -b2               - Replace \"\\special{BMF=file.bmf}\" with\n");
  printf("                        \"\\special{isoscale file.png, ####in ####in}\" (dviwin).\n");
  printf("    -q                - Suppress verbose messages (run quietly).\n");
  printf("    -c#-#             - Specify range of character codes to use in temperary\n");
  printf("                        PK fonts (e.g., -c32-127).\n");
  printf("    -p<str>           - Use \"str\" as the prefix of PK fonts\n");
#ifndef UNIX
  printf("                        (maxi 5 chars, default \"tmp\")\n\n");
  printf("The PK fonts generated will be put in the directory \"dpi888\".\n");
#else
  printf("                        (default \"{username}-{dviname}\").\n\n");
  printf("The PK fonts will be saved as:\n");
  printf("\t\"${CCPKPATH}/{prefix}###.888pk\".\n");
  printf("or\n");
  printf("\t\"/usr/local/lib/texmf/fonts/cct/pk/{prefix}###.888pk\".\n");
#endif
  exit(1);
}

void check_dvi_ext _P1(byte *,s)
{
  /* append .dvi to file name when necessary */
#ifndef UNIX
  register int i;

  i=strlen(s)-1;
  for (;i>=0 && s[i]!='.';i--);
  if (i<0 || s[i+1]=='\\' || s[i+1]=='/') strcat(s,".dvi");
#else
  if (strcmp(s+strlen(s)-4,".dvi")) strcat(s,".dvi");
#endif
}

void parse_cmdline _P2(int,argc, byte **,argv)
{
  register byte *p;
  byte s[PATH_MAX];
  register int i;

  if (argc<3) usage();

  s_in[0]=s_out[0]='\0';
  for (i=1; i<argc; i++) {
    if (argv[i][0]!='-') {
      if (s_in[0]) {
        if (s_out[0]) usage();
        strcpy(s_out,argv[i]);
      } else strcpy(s_in,argv[i]);
      continue;
    }
    switch (toupper(argv[i][1])) {
      case 'Q':		/* run quietly */
        quiet=1;
        break;
      case 'Y':		/* Yes to all questions */
        yes_flag=1;
        break;
      case 'M':	      	/* magnification */
	if ((mag=atol(argv[i]+2))<0) {
	  fprintf(stderr,"Invalid magnification!\n");
	  usage();
	}
	break;
      case 'R':         /* DPIX[xDPIY] */
	strlwr(strcpy(s,argv[i]+2));
	if ((p=strchr(s,'x'))==NULL) dpix=dpiy=atof(s);
	else {p[0]='\0'; dpix=atof(s); dpiy=atof(p+1);}
	if (dpix<=0.0 || dpix>4096 || dpiy<=0.0 || dpiy>4096) {
	    fprintf(stderr,"Invalid resolution!\n");
	    usage();
	}
	break;
      case 'F':		/* don't generate PK fonts */
        pk_flag=0;
        break;
      case 'B':
        bmf_flag=1+atoi(argv[i]+2);
        break;
      case 'C':
        if (sscanf(argv[i]+2, "%d-%d", &code0, &code1)!=2 ||
            code0<0 || code0>255 || code1<code0 || code1>255) {
	    fprintf(stderr,"Invalid range specification!\n");
	    usage();
        }
        ncodes=code1-code0+1;
        break;
      case 'P':
#ifdef UNIX
        strcpy(pk_name,argv[i]+2);
#else
        strncpy(pk_name,argv[i]+2,5);
        pk_name[5]='\0';
#endif
        break;
      default:
        fprintf(stderr,"Invalid option \"%s\"!\n",argv[i]);
        usage();
    }
  }

  if (dpix<=0) dpix=dpiy=300.0;
  if (!s_in[0] || !s_out[0]) usage();
  check_dvi_ext(s_in);
  check_dvi_ext(s_out);

  /* find PK font path & hzfont path */
#ifndef UNIX
#if 0
  strcpy(pkpath, fullpath(argv[0]));
  i=strlen(pkpath)-1;
  while (i>=0 && pkpath[i]!='/' && pkpath[i]!='\\' && pkpath[i]!=':') i--;
  pkpath[i+1]='\0';
  strcpy(hzpath,pkpath);
  strcat(pkpath,"PIXEL\\DPI$d");
  strcat(hzpath,"HZFONTS");
#else
  strcpy(hzpath, "c:\\ctex\\localtexmf\\cct\\fonts");
  strcpy(pkpath, "c:\\ctex\\localtexmf\\fonts\\pk\\modeless\\cct\\dpi$d");
#endif
  if ((p=getenv("CCPKPATH"))!=NULL) strcpy(pkpath,p);
  if ((p=getenv("CCHZPATH"))!=NULL) strcpy(hzpath,p);

  /* replace "$d" with "%d" */
  if ((p=strchr(pkpath,'$'))==NULL) ErrExit("Invalid CCPKPATH variable!");
  *p='%';

  if (pkpath[strlen(pkpath)-1]!='\\' && pkpath[strlen(pkpath)-1]!='/')
	  strcat(pkpath,"\\");
  if (hzpath[strlen(hzpath)-1]!='\\' && hzpath[strlen(hzpath)-1]!='/')
	  strcat(hzpath,"\\");

  sprintf(pkpath+strlen(pkpath),"%s%s.pk",pk_name,"%03d");
  strcat(pk_name,"%03d");
#else
  if (pk_name[0] == '\0') {
    /* form default pk_name ("uname-dviname") */
    struct passwd *pw=getpwuid(getuid());
    /* skip path part in "s_out" */
    p=s_out+strlen(s_out);
    while (p>s_out && *p!='/') p--;
    if (*p=='/') p++;
    /* remove ".dvi" from s_out for use in pk_name */
    strcpy(s,p);
    s[strlen(s)-4]='\0';
    sprintf(pk_name,"%s-%s", (pw==NULL) ? "foo":pw->pw_name, s);
  }

  strcpy(pkpath,"/usr/local/lib/texmf/fonts/cct/pk");
  strcpy(hzpath,"/usr/local/lib/texmf/fonts/cct/hzfonts");

  if ((p=getenv("CCPKPATH"))!=NULL) strcpy(pkpath,p);
  if ((p=getenv("CCHZPATH"))!=NULL) strcpy(hzpath,p);

  if (pkpath[strlen(pkpath)-1]!='/') strcat(pkpath,"/");
  if (hzpath[strlen(hzpath)-1]!='/') strcat(hzpath,"/");

  strcat(pkpath,pk_name);  strcat(pkpath,"%d.%dpk");
  strcat(pk_name,"%d");
#endif
}

void mkpath _P1(byte *,pathname)
/* make subdirectory chains for "pathname" */
{
#if !defined(UNIX)
  register byte *p;

  strupr(pathname);
  for (p=pathname; *p; p++) if (*p=='/') *p='\\';
  if ((p=strchr(pathname,':'))==NULL) p=pathname; else p++;
  do {
    if (*p=='\\') p++;
    if ((p=strchr(p,'\\'))==NULL) break;
    p[0]='\0';
#ifdef GO32
    mkdir(pathname, 0755);
#else
    mkdir(pathname);
#endif
    if (!quiet) fprintf(stderr,"mkdir %s\n",pathname);
    p[0]='\\';
  } while (1);
#endif
}

void *int_to_bytes _P2(void *,p, int,size)
/* returns the pointer to the buffer containing the results */
{
  register byte *p0, *p1;
  static byte buffer[4];

  if (byte_order==MSB) {		/* MSB first */
    memcpy(buffer, ((byte *)p)+sizeof(long)-size, size);
  } else {				/* LSB first */
    p0=buffer, p1=((byte *)p)+size-1;
    while (size--) *(p0++)=*(p1--);
  }
  return(buffer);
}

long bytes_to_int _P2(void *,buffer, int,size)
{
  register byte *p=buffer;
  long ll;

  switch (size) {
    case 0: return 0;
    case 1: return p[0];
    case 2: return p[1]+256l*p[0];
    case 3: return p[2]+256l*(p[1]+256l*p[0]);
    case 4: return p[3]+256l*(p[2]+256l*(p[1]+256l*p[0]));
    default:
      ll=0l;
      while (size--) ll=256l*ll+*(p++);
      return ll;
  }
}

void copybytes _P1(unsigned long,l)
{
  register unsigned n;

  /* For some reason (may be bad implementation of disk cache of Windows 95),
     the fseek function is very slow, so we avoid to use it when possible. */
/*  if (pass) {*/
    while (l) {
      n=(l>BUFFERSIZE) ? BUFFERSIZE:(int)l;
      fread(buffer,n,1,f_in);
      if (pass) fwrite(buffer,n,1,f_out);
      l-=(long)n;
    }
/*  } else fseek(f_in,l,SEEK_CUR);*/
}

static void push_ziti _VOID
{
  if (ziti_stack>=ziti_stack0+ZITI_STACK_DEPTH) ErrExit("Ziti stack overflow!");
  (ziti_stack++)->ziti=ziti;
}

static void pop_ziti _VOID
{
  if (ziti_stack<=ziti_stack0) ErrExit("Ziti stack underflow!");
  ziti=(--ziti_stack)->ziti;
}

static void set_ziti _P1(register int,c)
{
  register int i;

  i=toupper(c)-'@';
  if (i==-1) push_ziti(); else if (i==-2) pop_ziti(); else ziti=i;
}

long readnumber _P1(int,nbytes)
{
  byte p[4];

  fread(p,nbytes,1,f_in);
  return bytes_to_int(p,nbytes);
}

void writenumber _P2(unsigned long,value, int,nbytes)
{
  if (!pass) return;
  fwrite(int_to_bytes(&value,nbytes),nbytes,1,f_out);
}

void out_setchar _P1(int,charno)
{
  if (charno<=127) fputc(charno,f_out);
  else if (charno<=255) {fputc(dvi_set1,f_out); fputc(charno,f_out);}
  else {fputc(dvi_set1+1,f_out); writenumber((unsigned long)charno,2);}
}

void out_putchar _P1(int,charno)
{
  if (charno<=255) {fputc(dvi_put1,f_out); fputc(charno,f_out);}
  else {fputc(dvi_put1+1,f_out); writenumber((unsigned long)charno,2);}
}

void out_setfont _P1(int,fontno)
{
  if (fontno<=63) fputc(fontno+dvi_fnt_num_0,f_out);
  else if (fontno<=255) {fputc(dvi_fnt1,f_out); fputc(fontno,f_out);}
  else {fputc(dvi_fnt1+1,f_out); writenumber((unsigned long)fontno,2);}
  outfontno=fontno;
}

void out_special _P1(byte *,str)
{
  register int len=strlen(str);

  if (len<=255) {fputc(dvi_xxx1,f_out); fputc(len,f_out);}
  else {fputc(dvi_xxx1+1,f_out); writenumber((unsigned long)len,2);}
  fwrite(str,len,1,f_out);
}

void out_fontdef _P7(int,fontno, unsigned long,cksum,
                     unsigned long,ssize, unsigned long,dsize,
                     int,a, int,l, byte *,fontname)
{
  if (fontno<=255) {fputc(dvi_fnt_def1,f_out); fputc(fontno,f_out);}
  else {fputc(dvi_fnt_def1+1,f_out); writenumber((unsigned long)fontno,2);}
  writenumber(cksum,4);
  writenumber(ssize,4);
  writenumber(dsize,4);
  fputc(a, f_out); fputc(l, f_out);
  fwrite(fontname,a+l,1,f_out);
  outfontno=fontno;
}

fontdescr *lookupfont _P1(int,fontno)
/* find font info for a font. returns NULL if not found */
{
  fontdescr *fp;
  for (fp=froot; fp!=NULL; fp=fp->chain) if (fp->fontno==fontno) return(fp);
  return(NULL);
}

int cccomp _P2(const ccdescr huge *,cc1, const ccdescr huge *,cc2)
{
  register int i;

  i=cc1->group - cc2->group;
  if (i) return(i);
  i=cc1->charno - cc2->charno;
  if (i) return(i);
  return(cc1->fontno - cc2->fontno);
}

void map_cc _P3(int,ziti, int *,fontno, int *,charno)
{
  register int c;
  long i,j,k;
  ccdescr cc;

  cc.fontno=*fontno;
  cc.charno=*charno;
  cc.group=GetGroup(ziti,cc.charno);

  i=0; j=totalcc-1; c=0;
  if (!cccomp(cclist+i,&cc)) {k=i; goto cont;}
  if (!cccomp(cclist+j,&cc)) {k=j; goto cont;}
  do {
    k=(i+j)>>1;
    c=cccomp(cclist+k,&cc);
    if (c<0) i=k; else j=k;
  } while (c && j-i>1);
cont:
  if (c) ErrExit("Unexpected error (incomplete Chinese chars?).");
  *charno=((int)k)%ncodes+code0;
  *fontno=ccfontnumbers[(int)(k/ncodes)];
}

#define addchar(q,n) {		\
  register byte far *p,m;	\
  p=q->charset+n/8;		\
  m=(1<<(n&7));			\
  if (!(*p&m)) q->nchars++;	\
  *p|=m;			\
}

void do_chinese _P2(int,charno, int,is_putchar)
{
  register int i;
  int fontno,group;

  group=GetGroup(ziti,charno);

  if (maxccno<charno || charno<0) {
    if (!pass) if (!quiet) Warning("invalid char no.");
    return;
  }

  if (!pass) {
    ccfontdescr *cp=NULL, *cq;
    cq=croot;
    while (cq!=NULL) {
      cp=cq;
      if (cp->group==group && cp->fontno==infontno) goto found;
      cq=cq->chain;
    }

    if ((cq=malloc(sizeof(ccfontdescr)))==NULL) ErrExit("Insufficient memory!");
    if ((cq->charset=farmalloc((maxccno+8)/8))==NULL) ErrExit("Insufficient memory!");
    for (i=0; i<=(maxccno+8)/8; i++) cq->charset[i]=0;
    cq->fontno=infontno;
    cq->group=group;
    cq->nchars=0;
    cq->chain=NULL;
    if (croot==NULL) croot=cq; else cp->chain=cq;
found:
    addchar(cq,charno);
    return;
  }

  /* find output fontno & charno for the character */
  fontno=infontno;
  map_cc(ziti, &fontno, &charno);
  if (fontno!=outfontno) out_setfont(fontno);
  is_putchar ? out_putchar(charno):out_setchar(charno);
#if 0
		{ /* replace cc with a rule */
		  double dy=((1.0/1048576.0)/72.27)/
		      ((double)num/(double)den*1e-5/2.54);
                  fontdescr *q;
                  long ll;
                  q=lookupfont(infontno);
		  fputc(is_putchar ? 137:132,f_out);
		  ll=(long)(q->cksum*dy+0.5);
		  fwrite(int_to_bytes(&ll,4),4,1,f_out);
		  ll=q->ssize;
		  fwrite(int_to_bytes(&ll,4),4,1,f_out);
                }
#endif
}

void do_setputchar _P2(int,charno, int,is_putchar)
{
  if (infontno==ccdummy) {
    set_ziti(charno);
    return;
  }
  if (!hzmode) {
    if (pass) is_putchar ? out_putchar(charno):out_setchar(charno);
    return;
  }
  if (ccsave==-1) {
    ccsave=charno;
    return;
  }
  do_chinese((ccsave-33)*94+charno-33,is_putchar);
  ccsave=-1;
}

void do_setfont _P1(int,fontno)
{
  fontdescr *fp;

  infontno=fontno;
  if (fontno==ccdummy) return;
  /* check if a chinese font */
  if ((fp=lookupfont(fontno))!=NULL) {
    if (fp->is_cc) { 	/* Chinese special font */
      hzmode=1;
      return;
    }
  }
  hzmode=0;
  if (pass) out_setfont(fontno);
}

void do_bmf _P1(byte *,bmfname)
{
  byte s_bmf[PATH_MAX], s_img[PATH_MAX];
  FILE *f_pic;
  long bmf_dpix, bmf_dpiy, ll;
  int  bmf_width,bmf_height,bmf_xoff,bmf_yoff;
  double fac;
  register byte *p;
  byte b[16];

  if (!bmf_flag) return;

  /* get BMF file name */
  while (isspace(*bmfname)) bmfname++;
  strncpy(s_bmf,bmfname,255); s_bmf[255]='\0';
  strlwr(s_bmf);
  for (p=s_bmf; *p; p++) if (*p=='\\') *p='/';
  /* construct new image file name */
  strcpy(s_img, s_bmf);
  p=s_img+strlen(s_img)-1;
  while (p>s_img && *p!='.') p--;
  if (*p=='.' && p[1]!='\\' && p[1]!='/')
    strcpy(p+1, bmf_flag <= 2 ? "pcx" : "png");
  else
    strcat(s_img, bmf_flag <= 2 ? "pcx" : "png");

  if ((f_pic=fopen(s_bmf,"rb"))==NULL) {
/*    if (!quiet) {
      sprintf(buffer,"can't open \"%s\", try \"%s\".",s_bmf, s_img);
      Warning(buffer);
    }*/
    goto try_image;
  }
  if (fread(&bmf_dpix,4,1,f_pic)!=1) {
bmf_error:
    if (f_pic!=NULL) fclose(f_pic);
    if (!quiet) {
      sprintf(buffer,"invalid BMF file \"%s\", try \"%s\".", s_bmf, s_img);
      Warning(buffer);
    }
    goto try_image;
  }
  if (bmf_dpix!=0xffffffffl) fseek(f_pic,0l,SEEK_SET);
  /* read BMF heading */
  if (fread(b,16,1,f_pic)!=1) goto bmf_error;
  /* The BMF files are also in MSB format as the DVI and PK files,
     so "bytes_to_int" may be used to read integers */
  bmf_dpix  =bytes_to_int(b,4);
  bmf_dpiy  =bytes_to_int(b+4,4);
  bmf_width =(int)bytes_to_int(b+8,2);
  bmf_height=(int)bytes_to_int(b+10,2);
  bmf_xoff  =(int)bytes_to_int(b+12,2);
  bmf_yoff  =(int)bytes_to_int(b+14,2);
#if defined(UNIX) || defined(GO32)	/* better check sizeof(int)>2 ? */
  if (bmf_xoff>32767) bmf_xoff-=65536;
  if (bmf_yoff>32767) bmf_yoff-=65536;
#endif
  if (bmf_width<=0 || bmf_height<=0 ||
      bmf_dpix<=0 || bmf_dpiy<=0) goto bmf_error;
  fclose(f_pic);
  goto cont;

try_image:

  if ((f_pic=fopen(s_img,"rb"))==NULL) {
    if (!quiet) {
      sprintf(buffer,"can't open/read \"%s\" or \"%s\".", s_bmf, s_img);
      Warning(buffer);
    }
    return;
  }

  if (fread(b,16,1,f_pic)!=1) {
image_error:
    if (f_pic!=NULL) fclose(f_pic);
    if (!quiet) {
      sprintf(buffer,"invalid PCX file \"%s\".", s_img);
      Warning(buffer);
    }
    return;
  }

  if (b[0]!=10) goto image_error;
  bmf_dpix  = (b[12]+256l*b[13])*65536l;
  bmf_dpiy  = (b[14]+256l*b[15])*65536l;
  bmf_width = (b[8]+256*b[9]) - (b[4]+256*b[5]) + 1;
  bmf_height= (b[10]+256*b[11]) - (b[6]+256*b[7]) + 1;
  bmf_xoff  = 0 /* b[4]+256*b[5] */;
  bmf_yoff  = 0 /* b[6]+256*b[7] */;
  if (bmf_width<=0 || bmf_height<=0 ||
      bmf_dpix<=0 || bmf_dpiy<=0) goto image_error;
  fclose(f_pic);

cont:
  /* adjust dvi position */
  fac=den*2.54/(num*1e-5);
  fputc(dvi_push,f_out);	/* push */
  ll=-(long)((bmf_xoff*65536.0/bmf_dpix)*fac+0.5);
  fputc(dvi_right4,f_out);	/* right4 */
  writenumber(ll,4);
  ll=-(long)(((bmf_yoff+bmf_height)*65536.0/bmf_dpiy)*fac+0.5);
  fputc(dvi_down4,f_out);	/* down4 */
  writenumber(ll,4);

  if (bmf_flag!=2) { /* xdvi and emTeX */
    sprintf(buffer,"em:graph %s %0.4fin %0.4fin",strlwr(s_img),
                    (float)(bmf_width*65536.0/bmf_dpix),
                    (float)(bmf_height*65536.0/bmf_dpiy));
  } else { /* DVIWIN */
    /* use one of "isoscale", "anisoscale", and "center" keywords */
    sprintf(buffer,"isoscale %s, %0.4fin %0.4fin",strlwr(s_img),
                    (float)(bmf_width*65536.0/bmf_dpix),
                    (float)(bmf_height*65536.0/bmf_dpiy));
  }
  out_special(buffer);
  fputc(dvi_pop,f_out);	/* pop position */
  dvistack_adjust=1;
  return;
}

void do_special _P1(byte *,str)
{

  while (isspace(*str)) str++;

  if (!strnicmp("ZiTi=",str,5)) {
    set_ziti(str[5]);
    return;
  }

  /* Filter out other CCT \specials */
  if (!pass) return;

  if (!strnicmp("BMF=",str,4)) {
    do_bmf(str+4);
    return;
  }

/*  if (!strnicmp("COLOR=",str,6)) return;

  if (!strnicmp("DVI=",str,4)) return;*/

#if 0	/* obsolete now (handled by ctex.c) */
  /* process src specials (removing 'ctextemp_' prefix in filenames) */
  if (!strncmp(str, "src:", 4)) {
    char *p = str + 4, *q, *q0;
    int have_space = 0;
#ifdef DEBUG
    fprintf(stderr, "|%s| ==> ", str);
#endif
    while (isspace(*p)) p++;
    /* skip line number */
    while (isdigit(*p)) p++;
    /* skip possible ":col number" */
    if (*p == ':')
      while (isdigit(*(++p)));
    /* skip optional space(s) */
    if (isspace(*p)) {
      have_space = 1;
      while (isspace(*p)) p++;
    }
    /* ctex's temp filename shouldn't contain a path */
    if (!strnicmp(p, "ctextemp_", 9)) {
      if (have_space || !isdigit(p[9]))
	strcpy(p, p + 9);
      else {
	*(p++) = ' ';
	strcpy(p, p + 8);
      }
    }
    /* check file extension */
    q = (q0 = p + strlen(p)) - 1;
    while (q >= p && *q != '.' && *q != '/'
#ifndef UNIX
	   && *q != '\\' && *q != ':'
#endif
	  )
      q--;
    if (q < p || *q != '.')
      q = q0;
    if (*q == '\0' || !stricmp(q, ".tex")) {
      /* check if .ctx file exists */
      char ext_save[5];
      static struct stat buf;

      strcpy(ext_save, q);
      strcpy(q, ".ctx");
      if (stat(p, &buf)) {
	/* .ctx file does not exist, restore original name */
	strcpy(q, ext_save);
      }
    }
#ifdef DEBUG
    fprintf(stderr, "|%s|\n", str);
#endif
  }
#endif

  out_special(str);
}

void do_fontdef _P7(int,fontno, unsigned long,cksum, unsigned long,ssize,
		    unsigned long,dsize, int,a, int,l, byte *,fontname)
{
  fontdescr *fp=NULL, *fq;
  int is_cc;

  fontname[a+l]='\0';
  if (!stricmp("ccdummy",fontname)) {
    if (!postamble) ccdummy=fontno;
    return;
  }
  is_cc=!strnicmp("ccfnt",fontname,5);
  if (is_cc) {
    /* cksum is char height, scale it accordingly */
    cksum = (unsigned long)(cksum * ((double)ssize / dsize) + 0.5);
  }
  if (!pass && !postamble) {
    if (is_cc && maxccheight<(long)cksum) maxccheight=cksum;
    fq=froot;
    while (fq!=NULL) {fp=fq; fq=fq->chain;}
    if ((fq=malloc(sizeof(fontdescr)))==NULL) ErrExit("Insufficient memory!");
    fq->fontno=fontno;
    fq->cksum=cksum;
    fq->ssize=ssize;
    fq->chain=NULL;
    fq->is_cc=is_cc;
    if (froot==NULL) froot=fq; else fp->chain=fq;
  }
  if (pass && !is_cc) out_fontdef(fontno, cksum, ssize, dsize, a, l, fontname);
}

void sortcc _VOID
{
  long gap,i,j,n=totalcc;
  ccdescr huge *v1, huge *v2, wk;

  for (gap=n/2; gap>0; gap>>=1) {
    for (i=gap; i<n; i++) {
      v2=cclist+i;
      for (j=i-gap; j>=0; j-=gap) {
	v1=v2-gap;
	if (cccomp(v1,v2)<=0) break;
	wk=*v1; *v1=*v2; *v2=wk;
	v2=v1;
      }
    }
  }
}

void gather_cc _VOID
{
  ccfontdescr *cq,*cp;
  fontdescr *fp;
  ccdescr huge *charptr;
  register int i,j,k,m,n,b;

  nbrofccfonts=0;
  totalcc=0l;
  cq=croot;
  while (cq!=NULL) {
    nbrofccfonts++;
    totalcc+=cq->nchars;
    cq=cq->chain;
  }

#if 0
  if (!totalcc && !bmf_flag) {
    fprintf(stderr,"This file contains no Chinese character!\n");
    fprintf(stderr,"Deleting \"%s\" ...\n",s_out);
    fclose(f_out); f_out=NULL;
    fclose(f_in); f_in=NULL;
    unlink(s_out);
    exit(1);
  }
#endif

  /* Compute ss0 such that (ss0/ds0)*dpix*mag*0.001 = 888 */
  ss0=888.0*ds0/(dpix*mag*0.001);

  if (totalcc) {
    if ((cclist=farmalloc(totalcc*(long)sizeof(ccdescr)))==NULL)
      ErrExit("Insufficient memory!");
  }

  charptr=cclist; cq=croot;
  for (i=0; cq!=NULL; i++) {
    n=0;
    for (j=0; j<=(maxccno+8)/8; j++) {
      if ((b=cq->charset[j])!=0) for (k=0, m=1; k<8; k++, m<<=1) if (b&m) {
        charptr->fontno=cq->fontno;
        charptr->charno=j*8+k;
        charptr->group=cq->group;
        charptr++; n++;
      }
    }
    if (n!=cq->nchars) ErrExit("Unexpected error!");
    farfree(cq->charset);
    cp=cq->chain; farfree(cq); cq=cp;
  }
  croot=NULL;

  /* sort cclist according to (group,charno,fontno) */
  sortcc();

  /* findout appropriate output fontno for Chinese fonts */
  nbroftmpfonts=(int)((totalcc+ncodes-1)/ncodes);
  if (nbroftmpfonts) {
    if ((ccfontnumbers=malloc(sizeof(unsigned)*nbroftmpfonts))==NULL)
      ErrExit("Insufficient memory!");
  }
  k=0;
  for (i=0; i<nbroftmpfonts; i++) {
    do {
      fp=lookupfont(k);
      if (fp==NULL) break;
      if (fp->is_cc) break;
      k++;
    } while (1);
    ccfontnumbers[i]=k;
    k++;
  }
}

void out_tmpfontdefs _VOID
{
  register int i;
  byte s[PATH_MAX];
  unsigned long cksum;
  unsigned long ssize;
  unsigned long dsize;
  double fac;

  fac=((1.0/1048576.0)/72.27)/((double)num/(double)den*1e-5/2.54);
  dsize=(unsigned long)(ds0*1048576.0*fac+0.5);
  ssize=(unsigned long)(ss0*1048576.0*fac+0.5);;
  cksum=0l;

  for (i=0; i<nbroftmpfonts; i++) {
    sprintf(s,pk_name,i);
    out_fontdef(ccfontnumbers[i], cksum, ssize, dsize, 0, strlen(s), s);
  }
}

#if GENPK

/*====================================================================*/

static byte mask[]={0,1,3,7,15,31,63,127,255};  /* mask[i]=(1<<i)-1 */

static long *runcount=NULL,*rctptr; /* buffer to save the run counts */
static unsigned rctlen;
static int dyn_f;  /* value of the current dyn_f */
static int savnyb,savflg;
static byte *pk_out_buff0,*pk_out_buff;

byte *setbits _P3(byte *,p, int *,a, int,n)
{
  int i,j;
  if ((i=*a+n)<=8) {
    *a=i;
    j=mask[n]; if ((i=8-i)!=0) j<<=i;
    *p|=(j & 255);
    if (*a>7) {p++; *a=0;}
  } else {
    if (*a) {*(p++)|=mask[i=(8-(*a))]; n-=i;}
    memset(p,255,j=n>>3); p+=j;
    if (n&=7) {
      *p|=mask[n]<<(8-n);
      *a=n;
    } else *a=0;
  }
  return(p);
}

byte *skipbits _P3(byte *,p, int *,a, int,n)
{
  int i;

  if ((i=(*a)+n)<=8) {
    if ((*a=i)>7) {p++; *a=0;}
  } else {
    if (*a) {p++; n-=8-(*a);}
    p+=n>>3;
    *a=n&7;
  }
  return(p);
}

static void putnyble _P1(register int,nyble)
/* puts a nyble */
{
  if (savflg) {
    if ((unsigned)(pk_out_buff-pk_out_buff0)>=BUFFERSIZE)
      ErrExit("Insufficient buffer size!");
    *(pk_out_buff++)=nyble | (savnyb<<4);
    savflg=0;
  } else {
    savflg=1; savnyb=nyble;
  }
}

static void putpknum _P1(unsigned long,n)
/* sends an integer in packed form as a sequence of nybles */
{
  long i,l;
  int j,k;
  byte h[18];

  if (n<=(byte)dyn_f) putnyble((unsigned int)n); else {
    if ((i=(l=n-dyn_f-1)>>4)<(k=13-dyn_f)) {
      putnyble((int)i+dyn_f+1); putnyble((int)(l & 15));
    } else {
      i=n-(k<<4)-dyn_f+15; j=0;
      while (i>0) {
        h[j++]=(int)(i & 15); i>>=4;
      }
      j--;
      for (i=0;i<j;i++) putnyble(0);
      for (;j>=0;) putnyble(h[j--]);
    }
  }
}

static void putpkcount _P1(long,n)
/* the most significant bit<>0 ==> it's a repeat count otherwise it's a run count */
{
  long i,j;
  if ((j=n) & 0x80000000l) {
    if ((i=(j & 0x7fffffffl))==1) putnyble(15);
    else {putnyble(14); putpknum(i);}
  } else putpknum(j);
}

/* The array element len[dyn_f*8+k] is initialized to memorize the largest
   integer which can be represented by k+1 nybles if k<=1 or 2*k-1
   nybles if k>=2, with respect to the value of dyn_f. We can easily get
   the following expression for len[k]:
     len[dyn_f*8+0]=dyn_f; len[dyn_f*8+1]=(13-dyn_f)*16+dyn_f;
     len[dyn_f*8+k]=(13-dyn_f)*16+dyn_f-16+(16**k), for k=2,...,7 */
static
     unsigned long len[]={0l,208l,448l,4288l,65728l,1048768l,16777408l,268435648l,
			  1l,193l,433l,4273l,65713l,1048753l,16777393l,268435633l,
                          2l,178l,418l,4258l,65698l,1048738l,16777378l,268435618l,
                          3l,163l,403l,4243l,65683l,1048723l,16777363l,268435603l,
                          4l,148l,388l,4228l,65668l,1048708l,16777348l,268435588l,
                          5l,133l,373l,4213l,65653l,1048693l,16777333l,268435573l,
                          6l,118l,358l,4198l,65638l,1048678l,16777318l,268435558l,
                          7l,103l,343l,4183l,65623l,1048663l,16777303l,268435543l,
                          8l,88l,328l,4168l,65608l,1048648l,16777288l,268435528l,
                          9l,73l,313l,4153l,65593l,1048633l,16777273l,268435513l,
                          10l,58l,298l,4138l,65578l,1048618l,16777258l,268435498l,
                          11l,43l,283l,4123l,65563l,1048603l,16777243l,268435483l,
                          12l,28l,268l,4108l,65548l,1048588l,16777228l,268435468l,
                          13l,13l,253l,4093l,65533l,1048573l,16777213l,268435453l};
static unsigned long lenstats[14];
			/* contains the packed length (in nybles) for
                           all possible values of dyn_f (only the
                           packed integers are counted) */
static unsigned long baselen;
			/* length of nybles not counted in lenstats */
static unsigned long rasterlen;

static void putcnt _P1(long,l)
{
  register int i,j,n;
  long k;
  if (rctptr-runcount>=(long)rctlen) {
    long *p;
    rctlen+=256;
    if ((p=realloc(runcount, rctlen*sizeof(long)))==NULL) ErrExit("Insufficient memory!");
    rctptr=rctptr+(int)(p-runcount);
    runcount=p;
  }
  *(rctptr++)=l;
  if (l & 0x80000000l) {
    if ((k=(l & 0x7fffffffl))==1) k=0;
    baselen++;
  } else k=l;
  if (k>0) {
    n=0;
    for (j=0;j<14;j++) {
      i=0;
      while ((i<8) && (len[n+i]<(unsigned long)k)) i++;
      if (i<=1) lenstats[j]+=i+1; else lenstats[j]+=i+i-1;
      n+=8;
    }
  }
}

static void pkbitimage _P4(register byte *,source, byte *,dest,
			   int,width, int,height)
/* packs bit image data */
{
  register int s;
  int d,i,j,l,n,pxltype;

  if ( ((height*(long)width+7)>>3) > BUFFERSIZE )
    ErrExit("Insufficient buffer size!");

  if (width & 7) {
    n=width>>3;
    memcpy(dest,source,n);
    dest+=n; source+=n;
    memset(dest,0,((height*width+7)>>3)-n);
    d=j=0; s=128; i=n<<3;
    pxltype=((s & *source)!=0);
    do {
      l=0;
      do {
        l++;
        if ((++i>=width)) {
          i=0; j++;
          source++; s=128;
        } else if (!(s>>=1)) {s=128; source++;}
      } while ((j<height) && (pxltype==((*source & s)!=0)));
      if (!pxltype) dest=skipbits(dest,&d,l); else dest=setbits(dest,&d,l);
      pxltype=!pxltype;
    } while (j<height);
  } else memcpy(dest,source,(width*height)>>3);
}

#define SET_REPFLAG(j, f)	if (f) rep[j>>3] |= (128>>(j&7)); \
				else rep[j>>3] &= ~(128>>(j&7));
#define GET_REPFLAG(j)	( rep[j>>3] & (128>>(j&7)) )

static unsigned long pkchar _P6(byte *,matrix, byte *,dest,
				int,width, int,height,
                                int *,dyn_f0, int *,firstpxl)
/* packs a character whose bit image is given by matrix in the expanded form.
   This function uses the global runcount to store pixel run counts.
   The most significant bit of runcount[i] indicates if it's a repeat
   count (1) or a pixel run count (0). "*firstpxl" indicates
   if the first pixel dot is a black pixel (!=0) or a white pixel (0)
   This function returns the length (in bytes) of the result */
{
  int i,j;
  long u,l;
  register int m;
  unsigned n;
  register byte *p;
  byte *p1,c;
  byte *rep, f;

  if ((rep=malloc((height+7)/8))==NULL) {
    ErrExit("pkbitimage(): Insufficient memory size!");
  }

  if (runcount==NULL) {
    rctlen=RCTLEN;
    if ((runcount=malloc(rctlen*sizeof(long)))==NULL) ErrExit("Insufficient memory!");
  }

  rctptr=runcount; savflg=0;
  baselen=0l; n=(width+7)>>3; m=255-mask[8*n-width];
  memset(lenstats,0,sizeof(unsigned long)*14);
  /* find the repeat rows which will be marked by j-th bit of rep[]!=0 */
  memset(rep,0,(height+7)/8);
  p=matrix;
  for (j=1;j<height;j++) {
    f=!(memcmp(p,p1=p+n,n));
    if (f) {
      /* totally white or black lines are not considered repeating */
      c=(*p1 & 128) ? 0xff:0;
      for (p=p1;p<p1+(n-1);) if (*(p++)!=c) goto cont;
      if ((*p & m)!=(c & m)) goto cont;
      f=0;
    }
  cont:
    SET_REPFLAG(j,f);
    p=p1;
  }
  /* computes run counts */
  p1=matrix; p=p1; m=64; f=((*firstpxl=((*p) & 128))!=0); j=0; i=1; l=1;
  if (f) c=255; else c=0;
  while (j<height) {
    if ((((*p) & m)!=0)==f) l++; else {
      putcnt(l); l=1; f=!f; c=~c;
    }
    i++;
    if (!(m>>=1)) {
      p++;
      m=i; i=width-7;
      while ((m<i) && (*p==c)) {p++; m+=8; l+=8;}
      i=m;
      m=128;
    }
    if (i>=width) {
      i=j+1;
      do {
        ++j; p1+=n;
      } while (j<height && GET_REPFLAG(j));
      if (j>i) {
        putcnt((unsigned long)(j-i) | 0x80000000l);
	u=*(rctptr-2);
	if (u+l<width) {
	  *(rctptr-2)=*(rctptr-1);
	  *(rctptr-1)=u;
	}
      }
      i=0; p=p1; m=128;
    }
  }

  free(rep); rep=NULL;

  if (l) putcnt(l);
  /* find dyn_f who minimizes the packed data length */
  l=lenstats[0];
  dyn_f=0;
  for (i=1;i<=13;i++) {
    if (l>(long)lenstats[i]) {l=lenstats[i]; dyn_f=i;}
  }
  if ((l=(l+baselen+1)>>1)<(long)(rasterlen=((width*(long)height+7)>>3))) {
    pk_out_buff0=pk_out_buff=dest;
    for (i=0,j=(int)(rctptr-runcount);i<j;) putpkcount(runcount[i++]);
    if (savflg) putnyble(0);
    *dyn_f0=dyn_f;
    return(l);
  } else {
    dyn_f=14;
    pkbitimage(matrix,dest,width,height);
    *dyn_f0=dyn_f;
    return(rasterlen);
  }
}

/*====================================================================*/

#define f_tfm		f_in
#define f_pk		f_out

void pkclose _VOID
{
  register int i;

  if (f_pk==NULL) return;
  fputc(pk_post,f_pk);
  i=4-(int)(ftell(f_pk)&3);
  while (i--) fputc(pk_no_op, f_pk);
  fclose(f_pk);
  f_pk=NULL;
}

void pkopen _P1(int,pkno)
{
  unsigned long ll;
  long dsize, ssize;
  byte s[PATH_MAX];
#ifndef UNIX
  static int pkpath_made=0;

  if (!pkpath_made) {
    /* make necessary directories */
    sprintf(buffer,pkpath,(int)(ss0/ds0*dpix*mag*0.001+0.5),0);
    mkpath(buffer);
    pkpath_made=1;
  }
#endif

  pkclose();

  dsize=(unsigned long)(ds0*1048576.0+0.5);
  ssize=(unsigned long)(ss0*1048576.0+0.5);
#ifndef UNIX
  sprintf(s,pkpath,(int)(dpix*mag*0.001*ssize/(double)dsize+0.5),pkno);
#else
  sprintf(s,pkpath,pkno,(int)(dpix*mag*0.001*ssize/(double)dsize+0.5));
#endif
  if (!quiet) fprintf(stderr,"\rCreating \"%s\": ",s);
  if ((f_pk=fopen(s,"w+b"))==NULL) ErrExit("Cannot create the PK file!");
  fputc(pk_pre,f_pk);
  fputc(89,f_pk);
  fputc(0,f_pk);
  /* design size */
  fwrite(int_to_bytes(&dsize,4),4,1,f_pk);
  /* checksum */
  ll=0l;
  fwrite(int_to_bytes(&ll,4),4,1,f_pk);
  /* ll=(# of horizontal dots per pt)*65536 */
  ll=(long)(ss0/ds0*dpix*mag*0.001*(65536.0/72.27)+0.5);
  fwrite(int_to_bytes(&ll,4),4,1,f_pk);       /* hppp */
  /* ll=(# of vertical dots per pt)*65536 */
  ll=(long)(ss0/ds0*dpiy*mag*0.001*(65536.0/72.27)+0.5);
  fwrite(int_to_bytes(&ll,4),4,1,f_pk);       /* vppp */
}

void pkheading _P10(int,cc, int,width, int,height, unsigned long,l,
               	    double,tfm, double,ds, int,dyn_f, int,firstpxl,
               	    int,hoff, int, voff)
/* put character definition heading in a .PK file */
{
  int  f,dm;
  long tfm_wd;

  dm=(int)((tfm/72.27)*dpix*mag*0.001);
  tfm_wd=(long)((tfm/ds)*1048576.0);  /* tfm width (/ds*2^20) */
/*------- !!!!!! l is to be tested ----------------------*/
  l+=8;   /* short form */
  /* write the flag byte */
  if (l>=1024 || width>255 || height>255 ||
      dm>255  || abs(hoff)>127 || abs(voff)>127) {
    l+=5;
    f=(dyn_f<<4) | 4;
    if (firstpxl) f|=8;
    fputc(f,f_pk);
    fwrite(int_to_bytes(&l,2),2,1,f_pk);
    fputc(cc,f_pk);   /* char. code */
    fwrite(int_to_bytes(&tfm_wd,3),3,1,f_pk);	/* tfm width */
    fwrite(int_to_bytes(&dm,2),2,1,f_pk);		/* tfm width in pixel dots (dm) */
    fwrite(int_to_bytes(&width,2),2,1,f_pk);
    fwrite(int_to_bytes(&height,2),2,1,f_pk);	/* width & height */
    fwrite(int_to_bytes(&hoff,2),2,1,f_pk);		/* l_off */
    fwrite(int_to_bytes(&voff,2),2,1,f_pk);		/* t_off */
  } else {
    f=(dyn_f<<4) | (((int)(l & 1023))>>8);
    if (firstpxl) f|=8;
    fputc(f,f_pk);
    fputc((int)(l & 255),f_pk);
    fputc(cc,f_pk);				/* char. code */
    fwrite(int_to_bytes(&tfm_wd,3),3,1,f_pk);	/* tfm width */
    fputc(dm,f_pk);				/* tfm width in pixel dots (dm) */
    fputc(width,f_pk);
    fputc(height,f_pk);			/* width & height */
    hoff=hoff>=0 ? hoff:256+hoff;
    fputc(hoff,f_pk);				/* l_off */
    voff=voff>=0 ? voff:256+voff;
    fputc(voff,f_pk);				/* t_off */
  }
}

#if 0
/*???????????????????????????????????????????????????????????????????????*/
void tfmgen _P3(int,nbrchars, long *,charno, double,ds)			?
/* generates a .TFM file */						?
{									?
  register int i,j;							?
  int n,m,size;								?
  long  l;								?
  /* sizeused indicates if the corresponding size is used */		?
  int  sizeused[MaxiSize+1];    					?
  int len[12]={0,   /* lf, to be calculated */				?
               2,   /* lh */						?
               0,   /* bc, to be reset */				?
               127, /* ec, to be reset */				?
               2,   /* nw, to be calculated */				?
               2,   /* nh to be calculated */				?
               2,   /* nd to be calculated */				?
               1,   /* ni */						?
               0,   /* nl */						?
               0,   /* nk */						?
               0,   /* ne */						?
               7};  /* np */						?
									?
  cputs("Creating file \""); cputs(tfmname); cputs("\"...");		?
  openout(tfmname,&f_pk);						?
									?
  for (i=0;i<=MaxiSize;) sizeused[i++]=0;				?
  for (i=0;i<nbrchars;i++) {						?
    size=(int)(charno[i] & 255l);					?
    sizeused[size]=1;							?
  }									?
									?
  /* lengths */								?
  n=1; for (i=0;i<=MaxiSize;i++) if (sizeused[i]) n++;			?
  len[2]=code0;  	    /* bc */					?
  len[3]=nbrchars-1+code0;  /* ec */					?
  len[4]=len[5]=len[6]=n;						?
  m=6+len[1]+(len[3]-len[2]+1);						?
  for (i=4;i<12;) m+=len[i++];						?
  len[0]=m;								?
  for (i=0;i<12;i++) prbytes(&f_pk,(byte *)(&len[i]),2);		?
  /* header data */							?
  l=0; prbytes(&f_pk,(byte *)(&l),4);  /* chk_sum */			?
  l=(long)(ds*1048576.0);						?
  prbytes(&f_pk,(byte *)(&l),4);/* design size */			?
  /* char_info table */							?
  for (i=0;i<nbrchars;i++) {						?
    l=charno[i];							?
    size=(int)(l & 255l);						?
    m=0; for (j=0;j<=size;j++) if (sizeused[j]) m++;			?
    pbyte(&f_pk,m); pbyte(&f_pk,m*16+m); pbyte(&f_pk,0); pbyte(&f_pk,0);?
  }									?
  /* width table */							?
  l=0; prbytes(&f_pk,(byte *)(&l),4);					?
  for (i=0;i<=MaxiSize;i++) if (sizeused[i]) {				?
    l=(long)(1048576.0*width[i]/ds); prbytes(&f_pk,(byte *)(&l),4);	?
  }									?
  /* height table */							?
  l=0; prbytes(&f_pk,(byte *)(&l),4);					?
  for (i=0;i<=MaxiSize;i++) if (sizeused[i]) {				?
    l=(long)(1048576.0*height[i]/ds); prbytes(&f_pk,(byte *)(&l),4);	?
  }									?
  /* depth table */							?
  l=0; prbytes(&f_pk,(byte *)(&l),4);					?
  for (i=0;i<=MaxiSize;i++) if (sizeused[i]) {				?
    l=(long)(1048576.0*(height[i]*v_space_factor[i])/ds);		?
    prbytes(&f_pk,(byte *)(&l),4);					?
  }									?
  /* italic correction table */						?
  l=0; prbytes(&f_pk,(byte *)(&l),4);					?
  /* lig-kern table */							?
  /* kern table */							?
  /* extensible char. words */						?
  /* parameters */							?
  l=0; for (i=0;i<len[11];i++) prbytes(&f_pk,(byte *)(&l),4);		?
									?
  closeout(&f_pk);							?
  putchar(8); putchar(8); putchar(8);					?
  puts("   ");								?
}									?
/* ???????????????????????????????????????????????????????????????????? */
#endif

struct {
  byte name[13];		/* font file name */
  int  fac;			/* scale factor*10000 */
} hzfonttable[NbrOfGroup];

static byte huge *ptrtab=NULL;

#if !defined(UNIX) && !defined(GO32) && !defined(WIN32)

static void near font_driver(union REGS *regs,struct SREGS *sregs)
{
#define INTR_NO	0x60
  long far *intrvec=(void far *)(4l*INTR_NO);
  long intsave;
  void far fontdrv();

  /* save old int 60H */
  intsave=*intrvec;
  /* point int60 to the driver */
  *intrvec=(long)fontdrv;
  /* call the driver */
  int86x(INTR_NO,regs,regs,sregs);
  /* restore int 60H */
  *intrvec=intsave;
}

static int near pkfunc_init(byte far *fn,unsigned workbufflen,byte far *ptrtab)
{
  union REGS regs;
  struct SREGS sregs;

  regs.x.ax=0;						/* function 0 */
  regs.x.dx=FP_OFF(fn); sregs.ds=FP_SEG(fn);		/* font name */
  regs.x.bx=FP_OFF(ptrtab); sregs.es=FP_SEG(ptrtab);	/* ptrtab addr. */
  regs.x.cx=workbufflen;					/* work buffer size */
  font_driver(&regs,&sregs);
  return(regs.x.ax);
}

static int near pkfunc(byte far *matrix,int no,int w,int h,int mode,
                       byte far *work)
{
  union REGS regs;
  struct SREGS sregs;

  regs.x.ax=0x100 | mode;				/* function 1 */
  regs.x.si=no;						/* char no. */
  regs.x.bx=w; regs.x.cx=h;				/* width & height */
  regs.x.di=FP_OFF(matrix); sregs.es=FP_SEG(matrix);	/* bitmap buffer */
  regs.x.dx=FP_OFF(work); sregs.ds=FP_SEG(work);	/* work buffer */
  font_driver(&regs,&sregs);
  return(regs.x.ax);
}

static int near pkfunc_close _VOID
{
  union REGS regs;
  struct SREGS sregs;

  if (pkfunc_buffer!=NULL && pkfunc_buffer!=buffer) {
    free(pkfunc_buffer);
    pkfunc_buffer=NULL;
  }

  regs.x.ax=0x200;		/* function 2 */
  font_driver(&regs,&sregs);
  return(regs.x.ax);
}

#else  /* UNIX & DJGPP */

#include "fontdrvc.h"

#endif /* UNIX */

static void near getccfontsdef _VOID
/* gets parameters from hzpath+"ccfonts.def" */
{
  FILE *fp;
  register int i;
  char s[PATH_MAX], wkstr[PATH_MAX];
  double d;

  /* read file "ccfonts.def" */
  strcpy(s,hzpath); strcat(s,"ccfonts.def");
  if ((fp=fopen(s,"rt"))==NULL) {
    sprintf(wkstr,"**Error : file \"%s\" not found.",s);
    ErrExit(wkstr);
  }

  /* read file name containing normal Chinese characters */
  for (i=0;i<NbrOfGroup;i++) {
    hzfonttable[i].fac=10000;
    if (!feof(fp)) {
      if (fscanf(fp,"%s %lf",s,&d)==2) {
        strncpy(hzfonttable[i].name,s,12);
        hzfonttable[i].name[12]='\0';
        if (d>0.5 && d<2.0) hzfonttable[i].fac=(int)(d*10000.0+0.5);
        continue;
      }
    }
    hzfonttable[i].name[0]='\0';
  }
  fclose(fp);
}

static void near ccheading _P1(int,group)
{
  register int i, j;
  byte wkstr[PATH_MAX];
  FILE *fp;
  unsigned buffersize;

  i=-1;
  if (!group) {		/* user defined font */
    /* first look for the font file by changing the extension of "dviname" to ".pz" */
    strcpy(wkstr,s_in);
    j=strlen(wkstr)-1;
    for (;j>=0 && wkstr[j]!='.';j--);
    if (j<0 || wkstr[j+1]=='\\' || wkstr[j+1]=='/') strcat(wkstr,".pz");
    else strcpy(wkstr+j,".pz");
    if ((fp=fopen(wkstr,"rb"))!=NULL) {i=0; fclose(fp);}
  }

  if (i) {
    strcpy(wkstr,hzpath);
    strcat(wkstr,hzfonttable[group].name);
#ifndef UNIX
    strlwr(wkstr);
#endif
  }
#if !defined(UNIX) && !defined(GO32)
  pkfunc_buffer=buffer;
  buffersize=BUFFERSIZE;
#else
  if (pkfunc_buffer!=NULL && pkfunc_buffer!=buffer) {
    free(pkfunc_buffer);
    pkfunc_buffer=NULL;
  }
  buffersize=(int)(maxccheight*((1.0/1048576.0)/72.27*dpiy*0.001*mag)+0.5);
  buffersize=(int)((buffersize*(long)hzfonttable[group].fac+5000l)/10000l);
  buffersize*=128;
  if (buffersize<=BUFFERSIZE) {
    buffersize=BUFFERSIZE;
    pkfunc_buffer=buffer;
  } else {
    pkfunc_buffer=malloc(buffersize);
    if (pkfunc_buffer==NULL) ErrExit("Insufficient memory!");
  }
#endif
  i=pkfunc_init((void far *)wkstr,buffersize,(void far *)ptrtab);
  if (i) {
    sprintf(wkstr,"FONTDRV: cannot open font \"%s%s\", error code=%d.",
            hzpath,hzfonttable[group].name,i);
    ErrExit(wkstr);
  }
  return;
}

void genfonts _VOID
{
  register int len, pkcharno, pkfontno;
  int i, v_off, h_off, width=0, height=0, dyn_f, firstpxl;
  long ic;
  ccdescr cc;
  int fontno,charno,group;
  double tfmwidth=0.0;
  double dviratiox,dviratioy,tfmratiox;
  byte far *rasterbuffer=NULL;
  int pkfunc_open_flag=0;
  fontdescr far *fp;

  if (!nbroftmpfonts) return;

  if ((ptrtab=farmalloc((HanZi1-HanZi0+1)*4l))==NULL)
    ErrExit("Insufficient memory!");

  getccfontsdef();

  dviratiox=(double)num/(double)den*1e-5/2.54*dpix*0.001*mag;
  dviratioy=(1.0/1048576.0)/72.27*dpiy*0.001*mag;
  tfmratiox=(1.0/65536.0);

  pkfontno=0;
  pkcharno=code0;
  fontno=-1; group=-1; fp=NULL, h_off=v_off=0;
  for (ic=0; ic<totalcc; ic++) {
#if !defined(UNIX) && !defined(GO32) && !defined(__MINGW32__)
    if (!quiet) fprintf(stderr,"%-6ld%c%c%c%c%c%c",ic,8,8,8,8,8,8);
#else
    if (!quiet) fprintf(stderr,"%-6d%c%c%c%c%c%c",ic,8,8,8,8,8,8);
#endif
    cc=cclist[ic];
    if (fontno!=cc.fontno || group!=cc.group) {
      int w0, h0;
      if (fontno!=cc.fontno) {
        if ((fp=lookupfont(fontno=cclist[ic].fontno))==NULL) {
          ErrExit("Unexpected error!");
        }
        fontno=cc.fontno;
      }
      w0=(int)(fp->ssize*dviratiox+0.5);
      h0=(int)(fp->cksum*dviratioy+0.5);
      width=(int)((w0*(long)hzfonttable[cc.group].fac+5000l)/10000l);
      if (!width) width=1;
      height=(int)((h0*(long)hzfonttable[cc.group].fac+5000l)/10000l);
      if (!height) height=1;
      h_off=(width-w0+1)/2;
      v_off=(height-(is_ky ? 1:height/5))-(width-w0+1)/2;
      tfmwidth=fp->ssize*tfmratiox;
      if (rasterbuffer!=NULL) farfree(rasterbuffer);
      if ((rasterbuffer=farmalloc(((width+7)/8)*(long)height))==NULL) {
        ErrExit("Insufficient memory!");
      }
      if (cc.group!=group) {
        if (pkfunc_open_flag) pkfunc_close();
        ccheading(group=cc.group);
        pkfunc_open_flag=1;
      }
    }
    if (f_pk==NULL) pkopen(pkfontno++);
    charno=cc.charno;
#if 0	/* modified on Jan 1997 */
    if (group>1) charno-= (!(group&1)) ? hanzi0:hanzi1;
#else
    if (group!=1) charno-= (!(group&1)) ? hanzi0:hanzi1;
#endif
    if ((i=pkfunc(rasterbuffer,charno,width,height,6,pkfunc_buffer))!=0) {
      sprintf(buffer,"failed to unpack the character. Error code=%d, char code=%d",i,(cc.charno/94+1)*100+(cc.charno%94+1));
      ErrExit(buffer);
    }
    len=(int)pkchar(rasterbuffer,buffer,width,height,&dyn_f,&firstpxl);
    pkheading(pkcharno,width,height,(unsigned long)len,tfmwidth,ss0,
              dyn_f,firstpxl, h_off, v_off);
    fwrite(buffer,len,1,f_pk);
    if (++pkcharno>code1) {pkclose(); pkcharno=code0;}
  }
  if (pkfunc_open_flag) pkfunc_close();
  if (rasterbuffer!=NULL) farfree(rasterbuffer);
  if (!quiet) fprintf(stderr,"\n");

  pkclose();	/* to make sure that the last PK file is closed */
}
#endif

int main _P2(int,argc, char **,argv)
{
  int i, b;
  byte *p, s[PATH_MAX];
  long ll, cksum, ssize, dsize;

  /* determines the byte order */
  ll=0x4321;
  if (*((byte *)&ll)==0x21) byte_order=LSB; else byte_order=MSB;

  parse_cmdline(argc,(byte **)argv);

  if ((buffer=malloc(BUFFERSIZE))==NULL) {
    fprintf(stderr,"Insufficient memory!");
    exit(1);
  }

  if ((f_in=fopen(s_in,"rb"))==NULL) {
    fprintf(stderr,"Can't open input file \"%s\"!\n",s_in);
    usage();
  }
  setvbuf(f_in,NULL,_IOFBF,30*1024);

  if (!quiet) fprintf(stderr,"\n%s\n", version);

  fread(buffer,15,1,f_in);
  if (buffer[0]!=dvi_pre || buffer[1]!=2) ErrExit("Invalid input file!");
  i=buffer[14];
  fread(buffer+15,i,1,f_in); buffer[15+i]='\0';
  if (!quiet) fprintf(stderr,"'%s'\n",buffer+15);

  is_ky=!strncmp(buffer+15,"Mer2Dvi",7);
  if (is_ky) {
    hanzi0+=5*94; hanzi1+=5*94; hanzi2+=5*94; maxccno+=5*94;
  }

  num=bytes_to_int(buffer+2,4);
  den=bytes_to_int(buffer+6,4);
  if (!mag) {
    mag=bytes_to_int(buffer+10,4);
  } else {
    memcpy(buffer+10,int_to_bytes(&mag,4),4);
  }

  if (!stricmp(s_in,s_out)) ErrExit("Input and output file can't be the same!");
  if (!yes_flag) {
    if ((f_out=fopen(s_out,"rb"))!=NULL) {
      fclose(f_out);
      f_out = NULL;
      fprintf(stderr,"Overwrite file \"%s\" (y/n) ? ",s_out);
      fgets(s, sizeof(s), stdin);
      for (p=s; isspace(*p); p++);
      if (*p!='y' && *p!='Y') ErrExit("Abort!");
    }
  }
  if ((f_out=fopen(s_out,"w+b"))==NULL) {
    fprintf(stderr,"Can't open output file \"%s\"!\n",s_out);
    fclose(f_in);
    exit(1);
  }
  setvbuf(f_out,NULL,_IOFBF,(unsigned)30*1024);

  fwrite(buffer,15+strlen(buffer+15),1,f_out);

  for (pass=0; pass<2; pass++) {
    fseek(f_in,ftell(f_out),SEEK_SET);
    /* add fontdefs for tmp fonts */
    if (pass) out_tmpfontdefs();
    ploc=-1l;
    postamble=0;
    ziti=-5;
    /* main loop */
    do {
      b=fgetc(f_in);
      if (b<=dvi_set_char_127) {	/* set_char_0 -- set_char_127 */
        do_setputchar(b,0);
        continue;
      }
      if (dvi_set1<=b && b<=dvi_set4) {	/* set_char_1 -- set_char_4 */
        do_setputchar((int)readnumber(b-dvi_set1+1),0);
        continue;
      }
      if (dvi_put1<=b && b<=dvi_put4) {		/* put_char_1 -- put_char_4 */
        do_setputchar((int)readnumber(b-dvi_put1+1),1);
        continue;
      }
      if (b==dvi_bop) {				/* bop */
        if (pass) {
          ll=ftell(f_out);
          fputc(b,f_out);
        }
        fread(buffer,4,11,f_in);
        if (pass) {
          memcpy(buffer+40,int_to_bytes(&ploc,4),4);
          fwrite(buffer,4,11,f_out);
          ploc=ll;
        }
        ll=bytes_to_int(buffer,4);
	if (!quiet) fprintf(stderr,"Pass %d, page %-8d\r",pass+1,ll);
	outfontno=-1;
	infontno=-1;
	hzmode=0;
        continue;
      }
      if (dvi_fnt_num_0<=b && b<=dvi_fnt_num_63) {
      						/* fnt_num_0 -- fnt_num_63 */
        do_setfont(b-dvi_fnt_num_0);
        continue;
      }
      if (dvi_fnt1<=b && b<=dvi_fnt4) {		/* fnt1 -- fnt4 */
        do_setfont((int)readnumber(b-dvi_fnt1+1));
        continue;
      }
      if (dvi_xxx1<=b && b<=dvi_xxx4) {		/* xxx1 -- xxx4 */
        ll=readnumber(b-dvi_xxx1+1);
        if (ll>=BUFFERSIZE) {
          if (!quiet) Warning("\\special string too long (ignored)!");
          continue;
        }
        i=(int)ll;
        fread(buffer,i,1,f_in);
	buffer[i]='\0';
        do_special(buffer);
        continue;
      }
      if (dvi_fnt_def1<=b && b<=dvi_fnt_def4) {	/* fnt_def1 -- fnt_def4 */
        ll=readnumber(b-dvi_fnt_def1+1);
        cksum=readnumber(4);
        ssize=readnumber(4);
        dsize=readnumber(4);
        b=fgetc(f_in);
        i=fgetc(f_in);
        fread(buffer,b+i,1,f_in);
        do_fontdef((int)ll, cksum, ssize, dsize, b, i, buffer);
        continue;
      }
      if (b==dvi_post) {
        if (pass) {
          ll=ftell(f_out);
          fputc(b,f_out);
        }
        fread(buffer,4,7,f_in);
        if (pass) {
          int stackdepth;
          /* adjust last page location */
          memcpy(buffer,int_to_bytes(&ploc,4),4);
          /* adjust magnification */
          memcpy(buffer+12,int_to_bytes(&mag,4),4);
          /* adjust stack depth */
          stackdepth=buffer[24]*256+buffer[25]+dvistack_adjust;
          memcpy(buffer+24,int_to_bytes(&stackdepth,2),2);
	 /* output dvi_post body */
          fwrite(buffer,4,7,f_out);
          ploc=ll;
          /* add fontdefs for tmp fonts */
          out_tmpfontdefs();
        }
        postamble=1;
        continue;
      }
      if (b==dvi_post_post) {
        if (pass) fputc(b,f_out);
        fread(buffer,5,1,f_in);
        if (pass) {
          memcpy(buffer,int_to_bytes(&ploc,4),4);
          fwrite(buffer,5,1,f_out);
        }
        break;					/* exit the loop */
      }
      if (b>=dvi_undef) {
        if (!pass) if (!quiet) fprintf(stderr,"Undefined command: 0x%x\n",b);
        continue;
      }
      if (pass) fputc(b,f_out);
      copybytes((unsigned long)CommLen[b-dvi_set1]);
    } while (1);
    if (!pass) gather_cc();
  }

  i=(int)(ftell(f_out)&3);
  i=i ? 8-i:4;
  memset(buffer,223,i);
  fwrite(buffer,i,1,f_out);

  fclose(f_in); f_in=NULL;
  fclose(f_out); f_out=NULL;

  /* create tmp pk fonts */
#if !defined(UNIX) && !defined(GO32) && !defined(__MINGW32__)
  if (!quiet) fprintf(stderr,"Number of Chinese characters=%ld\n",totalcc);
#else
  if (!quiet) fprintf(stderr,"Number of Chinese characters=%d\n",totalcc);
#endif
  if (pk_flag) {
#if GENPK
    genfonts();
    if (!quiet) fprintf(stderr,"Number of PK fonts generated=%d\n",nbroftmpfonts);
#else
    fprintf(stderr,"Warning: this version of PATCHDVI doesn't allow creating PK fonts!\n");
#endif
  }

  return 0;
}
